/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.testsuite.oauth;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.nio.file.Files;
import java.security.Key;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.X509Certificate;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.ws.rs.core.Response;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.ContentBody;
import org.apache.http.entity.mime.content.FileBody;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.message.BasicNameValuePair;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Rule;
import org.junit.Test;
import org.keycloak.adapters.AdapterUtils;
import org.keycloak.adapters.authentication.JWTClientCredentialsProvider;
import org.keycloak.admin.client.resource.ClientAttributeCertificateResource;
import org.keycloak.admin.client.resource.ClientResource;
import org.keycloak.admin.client.resource.RealmResource;
import org.keycloak.authentication.AuthenticationFlowError;
import org.keycloak.common.util.Base64;
import org.keycloak.common.util.Base64Url;
import org.keycloak.common.util.BouncyIntegration;
import org.keycloak.common.util.KeyUtils;
import org.keycloak.common.util.KeycloakUriBuilder;
import org.keycloak.common.util.KeystoreUtil;
import org.keycloak.common.util.PemUtils;
import org.keycloak.common.util.Time;
import org.keycloak.common.util.UriUtils;
import org.keycloak.crypto.ECDSASignatureProvider;
import org.keycloak.crypto.SignatureSignerContext;
import org.keycloak.events.EventType;
import org.keycloak.jose.jwk.JSONWebKeySet;
import org.keycloak.jose.jws.JWSBuilder;
import org.keycloak.models.utils.KeycloakModelUtils;
import org.keycloak.protocol.oidc.OIDCAdvancedConfigWrapper;
import org.keycloak.representations.AccessToken;
import org.keycloak.representations.JsonWebToken;
import org.keycloak.representations.KeyStoreConfig;
import org.keycloak.representations.RefreshToken;
import org.keycloak.representations.idm.ClientRepresentation;
import org.keycloak.representations.idm.CredentialRepresentation;
import org.keycloak.representations.idm.EventRepresentation;
import org.keycloak.representations.idm.RealmRepresentation;
import org.keycloak.representations.idm.UserRepresentation;
import org.keycloak.testsuite.AbstractKeycloakTest;
import org.keycloak.testsuite.Assert;
import org.keycloak.testsuite.AssertEvents;
import org.keycloak.testsuite.admin.ApiUtil;
import org.keycloak.testsuite.arquillian.annotation.AuthServerContainerExclude;
import org.keycloak.testsuite.client.resources.TestApplicationResourceUrls;
import org.keycloak.testsuite.client.resources.TestOIDCEndpointsApplicationResource;
import org.keycloak.testsuite.util.ClientBuilder;
import org.keycloak.testsuite.util.ClientManager;
import org.keycloak.testsuite.util.OAuthClient;
import org.keycloak.testsuite.util.RealmBuilder;
import org.keycloak.testsuite.util.UserBuilder;
import org.keycloak.util.JsonSerialization;

@AuthServerContainerExclude(value={AuthServerContainerExclude.AuthServer.REMOTE})
public class ClientAuthSignedJWTTest
extends AbstractKeycloakTest {
    @Rule
    public AssertEvents events = new AssertEvents(this);
    private static String client1SAUserId;
    private static RealmRepresentation testRealm;
    private static ClientRepresentation app1;
    private static ClientRepresentation app2;
    private static ClientRepresentation app3;
    private static UserRepresentation defaultUser;
    private static UserRepresentation serviceAccountUser;

    @BeforeClass
    public static void beforeClientAuthSignedJWTTest() {
        BouncyIntegration.init();
    }

    @Override
    public void beforeAbstractKeycloakTest() throws Exception {
        super.beforeAbstractKeycloakTest();
    }

    @Override
    public void addTestRealms(List<RealmRepresentation> testRealms) {
        RealmBuilder realmBuilder = RealmBuilder.create().name("test").privateKey("MIICXAIBAAKBgQCrVrCuTtArbgaZzL1hvh0xtL5mc7o0NqPVnYXkLvgcwiC3BjLGw1tGEGoJaXDuSaRllobm53JBhjx33UNv+5z/UMG4kytBWxheNVKnL6GgqlNabMaFfPLPCF8kAgKnsi79NMo+n6KnSY8YeUmec/p2vjO2NjsSAVcWEQMVhJ31LwIDAQABAoGAfmO8gVhyBxdqlxmIuglbz8bcjQbhXJLR2EoS8ngTXmN1bo2L90M0mUKSdc7qF10LgETBzqL8jYlQIbt+e6TH8fcEpKCjUlyq0Mf/vVbfZSNaVycY13nTzo27iPyWQHK5NLuJzn1xvxxrUeXI6A2WFpGEBLbHjwpx5WQG9A+2scECQQDvdn9NE75HPTVPxBqsEd2z10TKkl9CZxu10Qby3iQQmWLEJ9LNmy3acvKrE3gMiYNWb6xHPKiIqOR1as7L24aTAkEAtyvQOlCvr5kAjVqrEKXalj0Tzewjweuxc0pskvArTI2Oo070h65GpoIKLc9jf+UA69cRtquwP93aZKtW06U8dQJAF2Y44ks/mK5+eyDqik3koCI08qaC8HYq2wVl7G2QkJ6sbAaILtcvD92ToOvyGyeE0flvmDZxMYlvaZnaQ0lcSQJBAKZU6umJi3/xeEbkJqMfeLclD27XGEFoPeNrmdx0q10Azp4NfJAY+Z8KRyQCR2BEG+oNitBOZ+YXF9KCpH3cdmECQHEigJhYg+ykOvr1aiZUMFT72HU0jnmQe2FVekuG+LJUt2Tm7GtMjTFoGpf0JwrVuZN39fOYAlo+nTixgeW7X8Y=").publicKey("MIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQCrVrCuTtArbgaZzL1hvh0xtL5mc7o0NqPVnYXkLvgcwiC3BjLGw1tGEGoJaXDuSaRllobm53JBhjx33UNv+5z/UMG4kytBWxheNVKnL6GgqlNabMaFfPLPCF8kAgKnsi79NMo+n6KnSY8YeUmec/p2vjO2NjsSAVcWEQMVhJ31LwIDAQAB").testEventListener();
        app1 = ClientBuilder.create().id(KeycloakModelUtils.generateId()).clientId("client1").attribute("jwt.credential.certificate", "MIICnTCCAYUCBgFPPLDaTzANBgkqhkiG9w0BAQsFADASMRAwDgYDVQQDDAdjbGllbnQxMB4XDTE1MDgxNzE3MjI0N1oXDTI1MDgxNzE3MjQyN1owEjEQMA4GA1UEAwwHY2xpZW50MTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAIUjjgv+V3s96O+Za9002Lp/trtGuHBeaeVL9dFKMKzO2MPqdRmHB4PqNlDdd28Rwf5Xn6iWdFpyUKOnI/yXDLhdcuFpR0sMNK/C9Lt+hSpPFLuzDqgtPgDotlMxiHIWDOZ7g9/gPYNXbNvjv8nSiyqoguoCQiiafW90bPHsiVLdP7ZIUwCcfi1qQm7FhxRJ1NiW5dvUkuCnnWEf0XR+Wzc5eC9EgB0taLFiPsSEIlWMm5xlahYyXkPdNOqZjiRnrTWm5Y4uk8ZcsD/KbPTf/7t7cQXipVaswgjdYi1kK2/zRwOhg1QwWFX/qmvdd+fLxV0R6VqRDhn7Qep2cxwMxLsCAwEAATANBgkqhkiG9w0BAQsFAAOCAQEAKE6OA46sf20bz8LZPoiNsqRwBUDkaMGXfnob7s/hJZIIwDEx0IAQ3uKsG7q9wb+aA6s+v7S340zb2k3IxuhFaHaZpAd4CyR5cn1FHylbzoZ7rI/3ASqHDqpljdJaFqPH+m7nZWtyDvtZf+gkZ8OjsndwsSBK1d/jMZPp29qYbl1+XfO7RCp/jDqro/R3saYFaIFiEZPeKn1hUJn6BO48vxH1xspSu9FmlvDOEAOz4AuM58z4zRMP49GcFdCWr1wkonJUHaSptJaQwmBwLFUkCbE5I1ixGMb7mjEud6Y5jhfzJiZMo2U8RfcjNbrN0diZl3jB6LQIwESnhYSghaTjNQ==").attribute("client_credentials.use_refresh_token", "true").authenticatorType("client-jwt").serviceAccountsEnabled(true).build();
        realmBuilder.client(app1);
        app2 = ClientBuilder.create().id(KeycloakModelUtils.generateId()).clientId("client2").directAccessGrants().serviceAccountsEnabled(true).redirectUris(OAuthClient.APP_ROOT + "/auth").attribute("jwt.credential.certificate", "MIICnTCCAYUCBgFPPQDGxTANBgkqhkiG9w0BAQsFADASMRAwDgYDVQQDDAdjbGllbnQxMB4XDTE1MDgxNzE4NTAwNVoXDTI1MDgxNzE4NTE0NVowEjEQMA4GA1UEAwwHY2xpZW50MTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMMw3PaBffWxgS2PYSDDBp6As+cNvv9kt2C4f/RDAGmvSIHPFev9kuQiKs3Oaws3ZsV4JG3qHEuYgnh9W4vfe3DwNwtD1bjL5FYBhPBFTw0lAQECYxaBHnkjHwUKp957FqdSPPICm3LjmTcEdlH+9dpp9xHCMbbiNiWDzWI1xSxC8Fs2d0hwz1sd+Q4QeTBPIBWcPM+ICZtNG5MN+ORfayu4X+Me5d0tXG2fQO//rAevk1i5IFjKZuOjTwyKB5SJIY4b8QTeg0g/50IU7Ht00Pxw6CK02dHS+FvXHasZlD3ckomqCDjStTBWdhJo5dST0CbOqalkkpLlCCbGA1yEQRsCAwEAATANBgkqhkiG9w0BAQsFAAOCAQEAUIMeJ+EAo8eNpCG/nXImacjrKakbFnZYBGD/gqeTGaZynkX+jgBSructTHR83zSH+yELEhsAy+3BfK4EEihp+PEcRnK2fASVkHste8AQ7rlzC+HGGirlwrVhWCdizNUCGK80DE537IZ7nmZw6LFG9P5/Q2MvCsOCYjRUvMkukq6TdXBXR9tETwZ+0gpSfsOxjj0ZF7ftTRUSzx4rFfcbM9fRNdVizdOuKGc8HJPA5lLOxV6CyaYIvi3y5RlQI1OHeS34lE4w9CNPRFa/vdxXvN7ClyzA0HMFNWxBN7pC/Ht/FbhSvaAagJBHg+vCrcY5C26Oli7lAglf/zZrwUPs0w==").authenticatorType("client-jwt").build();
        realmBuilder.client(app2);
        defaultUser = UserBuilder.create().id(KeycloakModelUtils.generateId()).username("test-user@localhost").password("password").build();
        realmBuilder.user(defaultUser);
        client1SAUserId = KeycloakModelUtils.generateId();
        serviceAccountUser = UserBuilder.create().id(client1SAUserId).username("service-account-" + app1.getClientId()).serviceAccountId(app1.getClientId()).build();
        realmBuilder.user(serviceAccountUser);
        testRealm = realmBuilder.build();
        testRealms.add(testRealm);
    }

    @Before
    public void recreateApp3() {
        app3 = ClientBuilder.create().id(KeycloakModelUtils.generateId()).clientId("client3").directAccessGrants().authenticatorType("client-jwt").build();
        Response resp = this.adminClient.realm("test").clients().create(app3);
        this.getCleanup().addClientUuid(ApiUtil.getCreatedId((Response)resp));
        resp.close();
    }

    @Test
    public void testServiceAccountAndLogoutSuccess() throws Exception {
        String client1Jwt = this.getClient1SignedJWT();
        OAuthClient.AccessTokenResponse response = this.doClientCredentialsGrantRequest(client1Jwt);
        org.junit.Assert.assertEquals((long)200L, (long)response.getStatusCode());
        AccessToken accessToken = this.oauth.verifyToken(response.getAccessToken());
        RefreshToken refreshToken = this.oauth.parseRefreshToken(response.getRefreshToken());
        this.events.expectClientLogin().client("client1").user(client1SAUserId).session(accessToken.getSessionState()).detail("token_id", accessToken.getId()).detail("refresh_token_id", refreshToken.getId()).detail("username", "service-account-client1").detail("client_auth_method", "client-jwt").assertEvent();
        org.junit.Assert.assertEquals((Object)accessToken.getSessionState(), (Object)refreshToken.getSessionState());
        client1Jwt = this.getClient1SignedJWT();
        OAuthClient.AccessTokenResponse refreshedResponse = this.doRefreshTokenRequest(response.getRefreshToken(), client1Jwt);
        AccessToken refreshedAccessToken = this.oauth.verifyToken(refreshedResponse.getAccessToken());
        RefreshToken refreshedRefreshToken = this.oauth.parseRefreshToken(refreshedResponse.getRefreshToken());
        org.junit.Assert.assertEquals((Object)accessToken.getSessionState(), (Object)refreshedAccessToken.getSessionState());
        org.junit.Assert.assertEquals((Object)accessToken.getSessionState(), (Object)refreshedRefreshToken.getSessionState());
        this.events.expectRefresh(refreshToken.getId(), refreshToken.getSessionState()).user(client1SAUserId).client("client1").detail("client_auth_method", "client-jwt").assertEvent();
        HttpResponse logoutResponse = this.doLogout(response.getRefreshToken(), this.getClient1SignedJWT());
        org.junit.Assert.assertEquals((long)204L, (long)logoutResponse.getStatusLine().getStatusCode());
        this.events.expectLogout(accessToken.getSessionState()).client("client1").user(client1SAUserId).removeDetail("redirect_uri").detail("client_auth_method", "client-jwt").assertEvent();
        response = this.doRefreshTokenRequest(response.getRefreshToken(), this.getClient1SignedJWT());
        org.junit.Assert.assertEquals((long)400L, (long)response.getStatusCode());
        org.junit.Assert.assertEquals((Object)"invalid_grant", (Object)response.getError());
        this.events.expectRefresh(refreshToken.getId(), refreshToken.getSessionState()).client("client1").user(client1SAUserId).removeDetail("token_id").removeDetail("updated_refresh_token_id").detail("client_auth_method", "client-jwt").error("invalid_token").assertEvent();
    }

    @Test
    public void testCodeToTokenRequestSuccess() throws Exception {
        this.oauth.clientId("client2");
        this.oauth.doLogin("test-user@localhost", "password");
        EventRepresentation loginEvent = this.events.expectLogin().client("client2").assertEvent();
        String code = (String)this.oauth.getCurrentQuery().get("code");
        OAuthClient.AccessTokenResponse response = this.doAccessTokenRequest(code, this.getClient2SignedJWT());
        org.junit.Assert.assertEquals((long)200L, (long)response.getStatusCode());
        this.oauth.verifyToken(response.getAccessToken());
        this.oauth.parseRefreshToken(response.getRefreshToken());
        this.events.expectCodeToToken((String)loginEvent.getDetails().get("code_id"), loginEvent.getSessionId()).client("client2").detail("client_auth_method", "client-jwt").assertEvent();
    }

    @Test
    public void testCodeToTokenRequestSuccessES256usingJwksUri() throws Exception {
        this.testCodeToTokenRequestSuccess("ES256", true);
    }

    @Test
    public void testCodeToTokenRequestSuccessES256usingJwks() throws Exception {
        this.testCodeToTokenRequestSuccess("ES256", false);
    }

    @Test
    public void testCodeToTokenRequestSuccessRS256usingJwksUri() throws Exception {
        this.testCodeToTokenRequestSuccess("RS256", true);
    }

    @Test
    public void testCodeToTokenRequestSuccessRS256usingJwks() throws Exception {
        this.testCodeToTokenRequestSuccess("RS256", false);
    }

    @Test
    public void testCodeToTokenRequestSuccessPS256usingJwksUri() throws Exception {
        this.testCodeToTokenRequestSuccess("PS256", true);
    }

    @Test
    public void testCodeToTokenRequestSuccessPS256usingJwks() throws Exception {
        this.testCodeToTokenRequestSuccess("PS256", false);
    }

    @Test
    public void testECDSASignature() throws Exception {
        this.testECDSASignatureLength(this.getClientSignedToken("ES256"), "ES256");
        this.testECDSASignatureLength(this.getClientSignedToken("ES384"), "ES384");
        this.testECDSASignatureLength(this.getClientSignedToken("ES512"), "ES512");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testCodeToTokenRequestSuccessES256Enforced() throws Exception {
        ClientResource clientResource = null;
        ClientRepresentation clientRep = null;
        try {
            clientResource = ApiUtil.findClientByClientId((RealmResource)this.adminClient.realm("test"), (String)"client2");
            clientRep = clientResource.toRepresentation();
            OIDCAdvancedConfigWrapper.fromClientRepresentation((ClientRepresentation)clientRep).setTokenEndpointAuthSigningAlg("ES256");
            clientResource.update(clientRep);
            this.testCodeToTokenRequestSuccess("ES256", true);
        }
        catch (Exception e) {
            Assert.fail();
        }
        finally {
            clientResource = ApiUtil.findClientByClientId((RealmResource)this.adminClient.realm("test"), (String)"client2");
            clientRep = clientResource.toRepresentation();
            OIDCAdvancedConfigWrapper.fromClientRepresentation((ClientRepresentation)clientRep).setTokenEndpointAuthSigningAlg(null);
            clientResource.update(clientRep);
        }
    }

    private void testECDSASignatureLength(String clientSignedToken, String alg) {
        String encodedSignature = clientSignedToken.split("\\.", 3)[2];
        byte[] signature = Base64Url.decode((String)encodedSignature);
        org.junit.Assert.assertEquals((long)ECDSASignatureProvider.ECDSA.valueOf((String)alg).getSignatureLength(), (long)signature.length);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private String getClientSignedToken(String alg) throws Exception {
        ClientRepresentation clientRepresentation = app2;
        ClientResource clientResource = this.getClient(testRealm.getRealm(), clientRepresentation.getId());
        clientRepresentation = clientResource.toRepresentation();
        try {
            KeyPair keyPair = this.setupJwksUrl(alg, clientRepresentation, clientResource);
            PublicKey publicKey = keyPair.getPublic();
            PrivateKey privateKey = keyPair.getPrivate();
            this.oauth.clientId("client2");
            this.oauth.doLogin("test-user@localhost", "password");
            String code = (String)this.oauth.getCurrentQuery().get("code");
            String clientSignedToken = this.createSignedRequestToken("client2", this.getRealmInfoUrl(), privateKey, publicKey, alg);
            OAuthClient.AccessTokenResponse response = this.doAccessTokenRequest(code, clientSignedToken);
            org.junit.Assert.assertEquals((long)200L, (long)response.getStatusCode());
            this.oauth.verifyToken(response.getAccessToken());
            this.oauth.openLogout();
            String string = clientSignedToken;
            return string;
        }
        finally {
            this.revertJwksUriSettings(clientRepresentation, clientResource);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void testCodeToTokenRequestSuccess(String algorithm, boolean useJwksUri) throws Exception {
        ClientRepresentation clientRepresentation = app2;
        ClientResource clientResource = this.getClient(testRealm.getRealm(), clientRepresentation.getId());
        clientRepresentation = clientResource.toRepresentation();
        try {
            KeyPair keyPair = useJwksUri ? this.setupJwksUrl(algorithm, clientRepresentation, clientResource) : this.setupJwks(algorithm, clientRepresentation, clientResource);
            PublicKey publicKey = keyPair.getPublic();
            PrivateKey privateKey = keyPair.getPrivate();
            this.oauth.clientId("client2");
            this.oauth.doLogin("test-user@localhost", "password");
            EventRepresentation loginEvent = this.events.expectLogin().client("client2").assertEvent();
            String code = (String)this.oauth.getCurrentQuery().get("code");
            OAuthClient.AccessTokenResponse response = this.doAccessTokenRequest(code, this.createSignedRequestToken("client2", this.getRealmInfoUrl(), privateKey, publicKey, algorithm));
            org.junit.Assert.assertEquals((long)200L, (long)response.getStatusCode());
            this.oauth.verifyToken(response.getAccessToken());
            this.oauth.parseRefreshToken(response.getRefreshToken());
            this.events.expectCodeToToken((String)loginEvent.getDetails().get("code_id"), loginEvent.getSessionId()).client("client2").detail("client_auth_method", "client-jwt").assertEvent();
        }
        finally {
            if (useJwksUri) {
                this.revertJwksUriSettings(clientRepresentation, clientResource);
            } else {
                this.revertJwksSettings(clientRepresentation, clientResource);
            }
        }
    }

    @Test
    public void testDirectGrantRequestSuccess() throws Exception {
        this.oauth.clientId("client2");
        OAuthClient.AccessTokenResponse response = this.doGrantAccessTokenRequest("test-user@localhost", "password", this.getClient2SignedJWT());
        org.junit.Assert.assertEquals((long)200L, (long)response.getStatusCode());
        AccessToken accessToken = this.oauth.verifyToken(response.getAccessToken());
        RefreshToken refreshToken = this.oauth.parseRefreshToken(response.getRefreshToken());
        this.events.expectLogin().client("client2").session(accessToken.getSessionState()).detail("grant_type", "password").detail("token_id", accessToken.getId()).detail("refresh_token_id", refreshToken.getId()).detail("username", "test-user@localhost").detail("client_auth_method", "client-jwt").removeDetail("code_id").removeDetail("redirect_uri").removeDetail("consent").assertEvent();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testSuccessWhenNoAlgSetInJWK() throws Exception {
        ClientRepresentation clientRepresentation = app2;
        ClientResource clientResource = this.getClient(testRealm.getRealm(), clientRepresentation.getId());
        clientRepresentation = clientResource.toRepresentation();
        try {
            String signingAlgorithm = "PS256";
            KeyPair keyPair = this.setupJwksUrl(signingAlgorithm, false, clientRepresentation, clientResource);
            PublicKey publicKey = keyPair.getPublic();
            PrivateKey privateKey = keyPair.getPrivate();
            this.oauth.clientId("client2");
            OAuthClient.AccessTokenResponse response = this.doGrantAccessTokenRequest("test-user@localhost", "password", this.createSignedRequestToken("client2", this.getRealmInfoUrl(), privateKey, publicKey, signingAlgorithm));
            org.junit.Assert.assertEquals((long)200L, (long)response.getStatusCode());
        }
        finally {
            this.revertJwksUriSettings(clientRepresentation, clientResource);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testSuccessDefaultAlgWhenNoAlgSetInJWK() throws Exception {
        ClientRepresentation clientRepresentation = app2;
        ClientResource clientResource = this.getClient(testRealm.getRealm(), clientRepresentation.getId());
        clientRepresentation = clientResource.toRepresentation();
        try {
            String signingAlgorithm = "RS256";
            KeyPair keyPair = this.setupJwksUrl(signingAlgorithm, false, clientRepresentation, clientResource);
            PublicKey publicKey = keyPair.getPublic();
            PrivateKey privateKey = keyPair.getPrivate();
            this.oauth.clientId("client2");
            OAuthClient.AccessTokenResponse response = this.doGrantAccessTokenRequest("test-user@localhost", "password", this.createSignedRequestToken("client2", this.getRealmInfoUrl(), privateKey, publicKey, signingAlgorithm));
            org.junit.Assert.assertEquals((long)200L, (long)response.getStatusCode());
            publicKey = keyPair.getPublic();
            privateKey = keyPair.getPrivate();
            this.oauth.clientId("client2");
            response = this.doGrantAccessTokenRequest("test-user@localhost", "password", this.createSignedRequestToken("client2", this.getRealmInfoUrl(), privateKey, publicKey, "PS256"));
            org.junit.Assert.assertEquals((long)400L, (long)response.getStatusCode());
            org.junit.Assert.assertEquals((Object)"Client authentication with signed JWT failed: Signature on JWT token failed validation", (Object)response.getErrorDescription());
            OIDCAdvancedConfigWrapper.fromClientRepresentation((ClientRepresentation)clientRepresentation).setTokenEndpointAuthSigningAlg("ES256");
            clientResource.update(clientRepresentation);
            response = this.doGrantAccessTokenRequest("test-user@localhost", "password", this.createSignedRequestToken("client2", this.getRealmInfoUrl(), privateKey, publicKey, "PS256"));
            org.junit.Assert.assertEquals((long)400L, (long)response.getStatusCode());
            org.junit.Assert.assertEquals((Object)"invalid signature algorithm", (Object)response.getErrorDescription());
        }
        finally {
            this.revertJwksUriSettings(clientRepresentation, clientResource);
            OIDCAdvancedConfigWrapper.fromClientRepresentation((ClientRepresentation)clientRepresentation).setTokenEndpointAuthSigningAlg(null);
            clientResource.update(clientRepresentation);
        }
    }

    @Test
    public void testDirectGrantRequestSuccessES256() throws Exception {
        this.testDirectGrantRequestSuccess("ES256");
    }

    @Test
    public void testDirectGrantRequestSuccessRS256() throws Exception {
        this.testDirectGrantRequestSuccess("RS256");
    }

    @Test
    public void testDirectGrantRequestSuccessPS256() throws Exception {
        this.testDirectGrantRequestSuccess("PS256");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void testDirectGrantRequestSuccess(String algorithm) throws Exception {
        ClientRepresentation clientRepresentation = app2;
        ClientResource clientResource = this.getClient(testRealm.getRealm(), clientRepresentation.getId());
        clientRepresentation = clientResource.toRepresentation();
        try {
            KeyPair keyPair = this.setupJwksUrl(algorithm, clientRepresentation, clientResource);
            PublicKey publicKey = keyPair.getPublic();
            PrivateKey privateKey = keyPair.getPrivate();
            this.oauth.clientId("client2");
            OAuthClient.AccessTokenResponse response = this.doGrantAccessTokenRequest("test-user@localhost", "password", this.createSignedRequestToken("client2", this.getRealmInfoUrl(), privateKey, publicKey, algorithm));
            org.junit.Assert.assertEquals((long)200L, (long)response.getStatusCode());
            AccessToken accessToken = this.oauth.verifyToken(response.getAccessToken());
            RefreshToken refreshToken = this.oauth.parseRefreshToken(response.getRefreshToken());
            this.events.expectLogin().client("client2").session(accessToken.getSessionState()).detail("grant_type", "password").detail("token_id", accessToken.getId()).detail("refresh_token_id", refreshToken.getId()).detail("username", "test-user@localhost").detail("client_auth_method", "client-jwt").removeDetail("code_id").removeDetail("redirect_uri").removeDetail("consent").assertEvent();
        }
        finally {
            this.revertJwksUriSettings(clientRepresentation, clientResource);
        }
    }

    @Test
    public void testClientWithGeneratedKeysJKS() throws Exception {
        this.testClientWithGeneratedKeys("JKS");
    }

    @Test
    public void testClientWithGeneratedKeysPKCS12() throws Exception {
        this.testClientWithGeneratedKeys("PKCS12");
    }

    private void testClientWithGeneratedKeys(String format) throws Exception {
        ClientRepresentation client = app3;
        UserRepresentation user = defaultUser;
        String keyAlias = "somekey";
        String keyPassword = "pwd1";
        String storePassword = "pwd2";
        KeyStoreConfig keyStoreConfig = new KeyStoreConfig();
        keyStoreConfig.setFormat(format);
        keyStoreConfig.setKeyPassword("pwd1");
        keyStoreConfig.setStorePassword("pwd2");
        keyStoreConfig.setKeyAlias("somekey");
        client = this.getClient(testRealm.getRealm(), client.getId()).toRepresentation();
        String certOld = (String)client.getAttributes().get("jwt.credential.certificate");
        byte[] keyStoreBytes = this.getClientAttributeCertificateResource(testRealm.getRealm(), client.getId()).generateAndGetKeystore(keyStoreConfig);
        ByteArrayInputStream keyStoreIs = new ByteArrayInputStream(keyStoreBytes);
        KeyStore keyStore = ClientAuthSignedJWTTest.getKeystore(keyStoreIs, "pwd2", format);
        keyStoreIs.close();
        client = this.getClient(testRealm.getRealm(), client.getId()).toRepresentation();
        X509Certificate x509Cert = (X509Certificate)keyStore.getCertificate("somekey");
        ClientAuthSignedJWTTest.assertCertificate(client, certOld, KeycloakModelUtils.getPemFromCertificate((X509Certificate)x509Cert));
        this.oauth.clientId(client.getClientId());
        PrivateKey privateKey = (PrivateKey)keyStore.getKey("somekey", "pwd1".toCharArray());
        KeyPair keyPair = new KeyPair(x509Cert.getPublicKey(), privateKey);
        OAuthClient.AccessTokenResponse response = this.doGrantAccessTokenRequest(user.getUsername(), ((CredentialRepresentation)user.getCredentials().get(0)).getValue(), this.getClientSignedJWT(keyPair, client.getClientId()));
        org.junit.Assert.assertEquals((long)200L, (long)response.getStatusCode());
        AccessToken accessToken = this.oauth.verifyToken(response.getAccessToken());
        RefreshToken refreshToken = this.oauth.parseRefreshToken(response.getRefreshToken());
        this.events.expectLogin().client(client.getClientId()).session(accessToken.getSessionState()).detail("grant_type", "password").detail("token_id", accessToken.getId()).detail("refresh_token_id", refreshToken.getId()).detail("username", user.getUsername()).detail("client_auth_method", "client-jwt").removeDetail("code_id").removeDetail("redirect_uri").removeDetail("consent").assertEvent();
    }

    @Test
    public void testUploadKeystoreJKS() throws Exception {
        this.testUploadKeystore("JKS", "client-auth-test/keystore-client1.jks", "clientkey", "storepass");
    }

    @Test
    public void testUploadKeystorePKCS12() throws Exception {
        this.testUploadKeystore("PKCS12", "client-auth-test/keystore-client2.p12", "clientkey", "pwd2");
    }

    @Test
    public void testUploadCertificatePEM() throws Exception {
        this.testUploadKeystore("Certificate PEM", "client-auth-test/certificate.pem", "undefined", "undefined");
    }

    @Test
    public void testUploadPublicKeyPEM() throws Exception {
        this.testUploadKeystore("Public Key PEM", "client-auth-test/publickey.pem", "undefined", "undefined");
    }

    @Test
    public void testUploadJWKS() throws Exception {
        this.testUploadKeystore("JSON Web Key Set", "clientreg-test/jwks.json", "undefined", "undefined");
    }

    private void testUploadKeystore(String keystoreFormat, String filePath, String keyAlias, String storePassword) throws Exception {
        ClientRepresentation client = this.getClient(testRealm.getRealm(), app3.getId()).toRepresentation();
        String certOld = (String)client.getAttributes().get("jwt.credential.certificate");
        URL fileUrl = this.getClass().getClassLoader().getResource(filePath);
        if (fileUrl == null) {
            throw new IOException("File not found: " + filePath);
        }
        File keystoreFile = new File(fileUrl.getFile());
        OAuthClient.AccessTokenResponse accessTokenResponse = this.oauth.doGrantAccessTokenRequest("master", "admin", "admin", null, "admin-cli", null);
        org.junit.Assert.assertEquals((long)200L, (long)accessTokenResponse.getStatusCode());
        String url = this.suiteContext.getAuthServerInfo().getContextRoot() + "/auth/admin/realms/" + testRealm.getRealm() + "/clients/" + client.getId() + "/certificates/jwt.credential/upload-certificate";
        FileBody fileBody = new FileBody(keystoreFile);
        HttpEntity entity = MultipartEntityBuilder.create().addPart("file", (ContentBody)fileBody).addTextBody("keystoreFormat", keystoreFormat).addTextBody("keyAlias", keyAlias).addTextBody("storePassword", storePassword).addTextBody("keyPassword", "undefined").build();
        HttpPost httpRequest = new HttpPost(url);
        httpRequest.setHeader("Authorization", "Bearer " + accessTokenResponse.getAccessToken());
        httpRequest.setEntity(entity);
        CloseableHttpClient httpClient = HttpClients.createDefault();
        HttpResponse httpResponse = httpClient.execute((HttpUriRequest)httpRequest);
        org.junit.Assert.assertEquals((long)200L, (long)httpResponse.getStatusLine().getStatusCode());
        client = this.getClient(testRealm.getRealm(), client.getId()).toRepresentation();
        if (keystoreFormat.equals("Public Key PEM")) {
            String pem = new String(Files.readAllBytes(keystoreFile.toPath()));
            String publicKeyNew = (String)client.getAttributes().get("jwt.credential.public.key");
            org.junit.Assert.assertEquals((String)"Certificates don't match", (Object)pem, (Object)publicKeyNew);
        } else if (keystoreFormat.equals("JSON Web Key Set")) {
            String publicKeyNew = (String)client.getAttributes().get("jwt.credential.public.key");
            PublicKey pk = KeycloakModelUtils.getPublicKey((String)publicKeyNew);
            Assert.assertNotNull((Object)pk);
        } else if (keystoreFormat.equals("Certificate PEM")) {
            String pem = new String(Files.readAllBytes(keystoreFile.toPath()));
            ClientAuthSignedJWTTest.assertCertificate(client, certOld, pem);
        } else {
            FileInputStream keystoreIs = new FileInputStream(keystoreFile);
            KeyStore keyStore = ClientAuthSignedJWTTest.getKeystore(keystoreIs, storePassword, keystoreFormat);
            ((InputStream)keystoreIs).close();
            String pem = KeycloakModelUtils.getPemFromCertificate((X509Certificate)((X509Certificate)keyStore.getCertificate(keyAlias)));
            ClientAuthSignedJWTTest.assertCertificate(client, certOld, pem);
        }
    }

    @Test
    public void testMissingClientAssertionType() throws Exception {
        LinkedList<NameValuePair> parameters = new LinkedList<NameValuePair>();
        parameters.add((NameValuePair)new BasicNameValuePair("grant_type", "client_credentials"));
        CloseableHttpResponse resp = this.sendRequest(this.oauth.getServiceAccountUrl(), parameters);
        OAuthClient.AccessTokenResponse response = new OAuthClient.AccessTokenResponse(resp);
        this.assertError(response, null, "invalid_client", "invalid_client_credentials");
    }

    @Test
    public void testInvalidClientAssertionType() throws Exception {
        LinkedList<NameValuePair> parameters = new LinkedList<NameValuePair>();
        parameters.add((NameValuePair)new BasicNameValuePair("grant_type", "client_credentials"));
        parameters.add((NameValuePair)new BasicNameValuePair("client_assertion_type", "invalid"));
        CloseableHttpResponse resp = this.sendRequest(this.oauth.getServiceAccountUrl(), parameters);
        OAuthClient.AccessTokenResponse response = new OAuthClient.AccessTokenResponse(resp);
        this.assertError(response, null, "invalid_client", "invalid_client_credentials");
    }

    @Test
    public void testMissingClientAssertion() throws Exception {
        LinkedList<NameValuePair> parameters = new LinkedList<NameValuePair>();
        parameters.add((NameValuePair)new BasicNameValuePair("grant_type", "client_credentials"));
        parameters.add((NameValuePair)new BasicNameValuePair("client_assertion_type", "urn:ietf:params:oauth:client-assertion-type:jwt-bearer"));
        CloseableHttpResponse resp = this.sendRequest(this.oauth.getServiceAccountUrl(), parameters);
        OAuthClient.AccessTokenResponse response = new OAuthClient.AccessTokenResponse(resp);
        this.assertError(response, null, "invalid_client", "invalid_client_credentials");
    }

    @Test
    public void testAssertionMissingIssuer() throws Exception {
        String invalidJwt = this.getClientSignedJWT(this.getClient1KeyPair(), null);
        LinkedList<NameValuePair> parameters = new LinkedList<NameValuePair>();
        parameters.add((NameValuePair)new BasicNameValuePair("grant_type", "client_credentials"));
        parameters.add((NameValuePair)new BasicNameValuePair("client_assertion_type", "urn:ietf:params:oauth:client-assertion-type:jwt-bearer"));
        parameters.add((NameValuePair)new BasicNameValuePair("client_assertion", invalidJwt));
        CloseableHttpResponse resp = this.sendRequest(this.oauth.getServiceAccountUrl(), parameters);
        OAuthClient.AccessTokenResponse response = new OAuthClient.AccessTokenResponse(resp);
        this.assertError(response, null, "invalid_client", "invalid_client_credentials");
    }

    @Test
    public void testAssertionUnknownClient() throws Exception {
        String invalidJwt = this.getClientSignedJWT(this.getClient1KeyPair(), "unknown-client");
        LinkedList<NameValuePair> parameters = new LinkedList<NameValuePair>();
        parameters.add((NameValuePair)new BasicNameValuePair("grant_type", "client_credentials"));
        parameters.add((NameValuePair)new BasicNameValuePair("client_assertion_type", "urn:ietf:params:oauth:client-assertion-type:jwt-bearer"));
        parameters.add((NameValuePair)new BasicNameValuePair("client_assertion", invalidJwt));
        CloseableHttpResponse resp = this.sendRequest(this.oauth.getServiceAccountUrl(), parameters);
        OAuthClient.AccessTokenResponse response = new OAuthClient.AccessTokenResponse(resp);
        this.assertError(response, "unknown-client", "invalid_client", "invalid_client_credentials");
    }

    @Test
    public void testAssertionDisabledClient() throws Exception {
        ClientManager.realm(this.adminClient.realm("test")).clientId("client1").enabled(false);
        String invalidJwt = this.getClient1SignedJWT();
        LinkedList<NameValuePair> parameters = new LinkedList<NameValuePair>();
        parameters.add((NameValuePair)new BasicNameValuePair("grant_type", "client_credentials"));
        parameters.add((NameValuePair)new BasicNameValuePair("client_assertion_type", "urn:ietf:params:oauth:client-assertion-type:jwt-bearer"));
        parameters.add((NameValuePair)new BasicNameValuePair("client_assertion", invalidJwt));
        CloseableHttpResponse resp = this.sendRequest(this.oauth.getServiceAccountUrl(), parameters);
        OAuthClient.AccessTokenResponse response = new OAuthClient.AccessTokenResponse(resp);
        this.assertError(response, "client1", "unauthorized_client", "client_disabled");
        ClientManager.realm(this.adminClient.realm("test")).clientId("client1").enabled(true);
    }

    @Test
    public void testAssertionUnconfiguredClientCertificate() throws Exception {
        class CertificateHolder {
            String certificate;

            CertificateHolder() {
            }
        }
        CertificateHolder backupClient1Cert = new CertificateHolder();
        backupClient1Cert.certificate = (String)ApiUtil.findClientByClientId((RealmResource)this.adminClient.realm("test"), (String)"client1").toRepresentation().getAttributes().get("jwt.credential.certificate");
        ClientManager.realm(this.adminClient.realm("test")).clientId("client1").updateAttribute("jwt.credential.certificate", null);
        String invalidJwt = this.getClient1SignedJWT();
        LinkedList<NameValuePair> parameters = new LinkedList<NameValuePair>();
        parameters.add((NameValuePair)new BasicNameValuePair("grant_type", "client_credentials"));
        parameters.add((NameValuePair)new BasicNameValuePair("client_assertion_type", "urn:ietf:params:oauth:client-assertion-type:jwt-bearer"));
        parameters.add((NameValuePair)new BasicNameValuePair("client_assertion", invalidJwt));
        CloseableHttpResponse resp = this.sendRequest(this.oauth.getServiceAccountUrl(), parameters);
        OAuthClient.AccessTokenResponse response = new OAuthClient.AccessTokenResponse(resp);
        this.assertError(response, "client1", "invalid_client", "client_credentials_setup_required");
        ClientManager.realm(this.adminClient.realm("test")).clientId("client1").updateAttribute("jwt.credential.certificate", backupClient1Cert.certificate);
    }

    @Test
    public void testAssertionInvalidSignature() throws Exception {
        String invalidJwt = this.getClientSignedJWT(this.getClient2KeyPair(), "client1");
        LinkedList<NameValuePair> parameters = new LinkedList<NameValuePair>();
        parameters.add((NameValuePair)new BasicNameValuePair("grant_type", "client_credentials"));
        parameters.add((NameValuePair)new BasicNameValuePair("client_assertion_type", "urn:ietf:params:oauth:client-assertion-type:jwt-bearer"));
        parameters.add((NameValuePair)new BasicNameValuePair("client_assertion", invalidJwt));
        CloseableHttpResponse resp = this.sendRequest(this.oauth.getServiceAccountUrl(), parameters);
        OAuthClient.AccessTokenResponse response = new OAuthClient.AccessTokenResponse(resp);
        this.assertError(response, "client1", "invalid_client", AuthenticationFlowError.CLIENT_CREDENTIALS_SETUP_REQUIRED.toString().toLowerCase());
    }

    @Test
    public void testAssertionExpired() throws Exception {
        String invalidJwt = this.getClient1SignedJWT();
        this.setTimeOffset(1000);
        LinkedList<NameValuePair> parameters = new LinkedList<NameValuePair>();
        parameters.add((NameValuePair)new BasicNameValuePair("grant_type", "client_credentials"));
        parameters.add((NameValuePair)new BasicNameValuePair("client_assertion_type", "urn:ietf:params:oauth:client-assertion-type:jwt-bearer"));
        parameters.add((NameValuePair)new BasicNameValuePair("client_assertion", invalidJwt));
        CloseableHttpResponse resp = this.sendRequest(this.oauth.getServiceAccountUrl(), parameters);
        OAuthClient.AccessTokenResponse response = new OAuthClient.AccessTokenResponse(resp);
        this.setTimeOffset(0);
        this.assertError(response, "client1", "invalid_client", "invalid_client_credentials");
    }

    @Test
    public void testParEndpointAsAudience() throws Exception {
        this.testEndpointAsAudience(this.oauth.getParEndpointUrl());
    }

    @Test
    public void testBackchannelAuthenticationEndpointAsAudience() throws Exception {
        this.testEndpointAsAudience(this.oauth.getBackchannelAuthenticationUrl());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void testEndpointAsAudience(String endpointUrl) throws Exception {
        ClientRepresentation clientRepresentation = app2;
        ClientResource clientResource = this.getClient(testRealm.getRealm(), clientRepresentation.getId());
        clientRepresentation = clientResource.toRepresentation();
        try {
            KeyPair keyPair = this.setupJwksUrl("PS256", clientRepresentation, clientResource);
            PublicKey publicKey = keyPair.getPublic();
            PrivateKey privateKey = keyPair.getPrivate();
            JsonWebToken assertion = this.createRequestToken(app2.getClientId(), this.getRealmInfoUrl());
            assertion.audience(new String[]{endpointUrl});
            LinkedList<NameValuePair> parameters = new LinkedList<NameValuePair>();
            parameters.add((NameValuePair)new BasicNameValuePair("grant_type", "client_credentials"));
            parameters.add((NameValuePair)new BasicNameValuePair("client_assertion_type", "urn:ietf:params:oauth:client-assertion-type:jwt-bearer"));
            parameters.add((NameValuePair)new BasicNameValuePair("client_assertion", this.createSignledRequestToken(privateKey, publicKey, "PS256", assertion)));
            try (CloseableHttpResponse resp = this.sendRequest(this.oauth.getServiceAccountUrl(), parameters);){
                OAuthClient.AccessTokenResponse response = new OAuthClient.AccessTokenResponse(resp);
                org.junit.Assert.assertNotNull((Object)response.getAccessToken());
            }
        }
        finally {
            this.revertJwksUriSettings(clientRepresentation, clientResource);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testInvalidAudience() throws Exception {
        ClientRepresentation clientRepresentation = app2;
        ClientResource clientResource = this.getClient(testRealm.getRealm(), clientRepresentation.getId());
        clientRepresentation = clientResource.toRepresentation();
        try {
            KeyPair keyPair = this.setupJwksUrl("PS256", clientRepresentation, clientResource);
            PublicKey publicKey = keyPair.getPublic();
            PrivateKey privateKey = keyPair.getPrivate();
            JsonWebToken assertion = this.createRequestToken(app2.getClientId(), this.getRealmInfoUrl());
            assertion.audience(new String[]{"https://as.other.org"});
            LinkedList<NameValuePair> parameters = new LinkedList<NameValuePair>();
            parameters.add((NameValuePair)new BasicNameValuePair("grant_type", "client_credentials"));
            parameters.add((NameValuePair)new BasicNameValuePair("client_assertion_type", "urn:ietf:params:oauth:client-assertion-type:jwt-bearer"));
            parameters.add((NameValuePair)new BasicNameValuePair("client_assertion", this.createSignledRequestToken(privateKey, publicKey, "PS256", assertion)));
            try (CloseableHttpResponse resp = this.sendRequest(this.oauth.getServiceAccountUrl(), parameters);){
                OAuthClient.AccessTokenResponse response = new OAuthClient.AccessTokenResponse(resp);
                org.junit.Assert.assertNull((Object)response.getAccessToken());
            }
        }
        finally {
            this.revertJwksUriSettings(clientRepresentation, clientResource);
        }
    }

    @Test
    public void testAssertionInvalidNotBefore() throws Exception {
        String invalidJwt = this.getClient1SignedJWT();
        this.setTimeOffset(-1000);
        LinkedList<NameValuePair> parameters = new LinkedList<NameValuePair>();
        parameters.add((NameValuePair)new BasicNameValuePair("grant_type", "client_credentials"));
        parameters.add((NameValuePair)new BasicNameValuePair("client_assertion_type", "urn:ietf:params:oauth:client-assertion-type:jwt-bearer"));
        parameters.add((NameValuePair)new BasicNameValuePair("client_assertion", invalidJwt));
        CloseableHttpResponse resp = this.sendRequest(this.oauth.getServiceAccountUrl(), parameters);
        OAuthClient.AccessTokenResponse response = new OAuthClient.AccessTokenResponse(resp);
        this.setTimeOffset(0);
        this.assertError(response, "client1", "invalid_client", "invalid_client_credentials");
    }

    @Test
    public void testAssertionReuse() throws Exception {
        String clientJwt = this.getClient1SignedJWT();
        OAuthClient.AccessTokenResponse response = this.doClientCredentialsGrantRequest(clientJwt);
        org.junit.Assert.assertEquals((long)200L, (long)response.getStatusCode());
        AccessToken accessToken = this.oauth.verifyToken(response.getAccessToken());
        Assert.assertNotNull((Object)accessToken);
        Assert.assertNull((Object)response.getError());
        response = this.doClientCredentialsGrantRequest(clientJwt);
        org.junit.Assert.assertEquals((long)400L, (long)response.getStatusCode());
        org.junit.Assert.assertEquals((Object)"invalid_client", (Object)response.getError());
    }

    @Test
    public void testMissingIdClaim() throws Exception {
        OAuthClient.AccessTokenResponse response = this.testMissingClaim("id");
        this.assertError(response, app1.getClientId(), "invalid_client", "invalid_client_credentials");
    }

    @Test
    public void testMissingIssuerClaim() throws Exception {
        OAuthClient.AccessTokenResponse response = this.testMissingClaim("issuer");
        this.assertError(response, null, "invalid_client", "invalid_client_credentials");
    }

    @Test
    public void testMissingSubjectClaim() throws Exception {
        OAuthClient.AccessTokenResponse response = this.testMissingClaim("subject");
        this.assertError(response, null, "invalid_client", "invalid_client_credentials");
    }

    @Test
    public void testMissingAudienceClaim() throws Exception {
        OAuthClient.AccessTokenResponse response = this.testMissingClaim("audience");
        this.assertError(response, app1.getClientId(), "invalid_client", "invalid_client_credentials");
    }

    @Test
    public void testMissingIssuedAtClaim() throws Exception {
        OAuthClient.AccessTokenResponse response = this.testMissingClaim("issuedAt");
        this.assertSuccess(response, app1.getClientId(), serviceAccountUser.getId(), serviceAccountUser.getUsername());
    }

    @Test
    public void testMissingExpirationClaim() throws Exception {
        OAuthClient.AccessTokenResponse response = this.testMissingClaim("expiration");
        this.assertSuccess(response, app1.getClientId(), serviceAccountUser.getId(), serviceAccountUser.getUsername());
        response = this.testMissingClaim(-11, "expiration");
        this.assertError(response, app1.getClientId(), "invalid_client", "invalid_client_credentials");
        response = this.testMissingClaim("expiration", "issuedAt");
        this.assertError(response, app1.getClientId(), "invalid_client", "invalid_client_credentials");
    }

    @Test
    public void testMissingNotBeforeClaim() throws Exception {
        OAuthClient.AccessTokenResponse response = this.testMissingClaim("notBefore");
        this.assertSuccess(response, app1.getClientId(), serviceAccountUser.getId(), serviceAccountUser.getUsername());
    }

    private OAuthClient.AccessTokenResponse testMissingClaim(String ... claims) throws Exception {
        return this.testMissingClaim(0, claims);
    }

    private OAuthClient.AccessTokenResponse testMissingClaim(int tokenTimeOffset, String ... claims) throws Exception {
        CustomJWTClientCredentialsProvider jwtProvider = new CustomJWTClientCredentialsProvider();
        jwtProvider.setupKeyPair(this.getClient1KeyPair());
        jwtProvider.setTokenTimeout(10);
        for (String claim : claims) {
            jwtProvider.enableClaim(claim, false);
        }
        Time.setOffset((int)tokenTimeOffset);
        String jwt = jwtProvider.createSignedRequestToken(app1.getClientId(), this.getRealmInfoUrl());
        Time.setOffset((int)0);
        return this.doClientCredentialsGrantRequest(jwt);
    }

    private void assertError(OAuthClient.AccessTokenResponse response, String clientId, String responseError, String eventError) {
        org.junit.Assert.assertEquals((long)400L, (long)response.getStatusCode());
        org.junit.Assert.assertEquals((Object)responseError, (Object)response.getError());
        this.events.expectClientLogin().client(clientId).session((String)null).clearDetails().error(eventError).user((String)null).assertEvent();
    }

    private void assertSuccess(OAuthClient.AccessTokenResponse response, String clientId, String userId, String userName) {
        org.junit.Assert.assertEquals((long)200L, (long)response.getStatusCode());
        AccessToken accessToken = this.oauth.verifyToken(response.getAccessToken());
        RefreshToken refreshToken = this.oauth.parseRefreshToken(response.getRefreshToken());
        this.events.expectClientLogin().client(clientId).user(userId).session(accessToken.getSessionState()).detail("token_id", accessToken.getId()).detail("refresh_token_id", refreshToken.getId()).detail("username", userName).detail("client_auth_method", "client-jwt").assertEvent();
    }

    private static void assertCertificate(ClientRepresentation client, String certOld, String pem) {
        pem = PemUtils.removeBeginEnd((String)pem);
        String certNew = (String)client.getAttributes().get("jwt.credential.certificate");
        org.junit.Assert.assertNotEquals((String)"The old and new certificates shouldn't match", (Object)certOld, (Object)certNew);
        org.junit.Assert.assertEquals((String)"Certificates don't match", (Object)pem, (Object)certNew);
    }

    @Test
    public void testCodeToTokenRequestFailureRS256() throws Exception {
        this.testCodeToTokenRequestFailure("RS256", "invalid_client", "client_credentials_setup_required");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testCodeToTokenRequestFailureES256Enforced() throws Exception {
        ClientResource clientResource = null;
        ClientRepresentation clientRep = null;
        try {
            clientResource = ApiUtil.findClientByClientId((RealmResource)this.adminClient.realm("test"), (String)"client2");
            clientRep = clientResource.toRepresentation();
            OIDCAdvancedConfigWrapper.fromClientRepresentation((ClientRepresentation)clientRep).setTokenEndpointAuthSigningAlg("ES256");
            clientResource.update(clientRep);
            this.testCodeToTokenRequestFailure("RS256", "invalid_client", "invalid_client_credentials");
        }
        catch (Exception e) {
            Assert.fail();
        }
        finally {
            clientResource = ApiUtil.findClientByClientId((RealmResource)this.adminClient.realm("test"), (String)"client2");
            clientRep = clientResource.toRepresentation();
            OIDCAdvancedConfigWrapper.fromClientRepresentation((ClientRepresentation)clientRep).setTokenEndpointAuthSigningAlg(null);
            clientResource.update(clientRep);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void testCodeToTokenRequestFailure(String algorithm, String error, String description) throws Exception {
        ClientRepresentation clientRepresentation = app2;
        ClientResource clientResource = this.getClient(testRealm.getRealm(), clientRepresentation.getId());
        clientRepresentation = clientResource.toRepresentation();
        try {
            KeyPair keyPair = this.setupJwksUrl(algorithm, clientRepresentation, clientResource);
            PublicKey publicKey = keyPair.getPublic();
            PrivateKey privateKey = keyPair.getPrivate();
            this.oauth.clientId("client2");
            this.oauth.doLogin("test-user@localhost", "password");
            EventRepresentation loginEvent = this.events.expectLogin().client("client2").assertEvent();
            String code = (String)this.oauth.getCurrentQuery().get("code");
            OAuthClient.AccessTokenResponse response = this.doAccessTokenRequest(code, this.getClient2SignedJWT());
            org.junit.Assert.assertEquals((long)400L, (long)response.getStatusCode());
            org.junit.Assert.assertEquals((Object)error, (Object)response.getError());
            this.events.expect(EventType.CODE_TO_TOKEN_ERROR).client("client2").session((String)null).clearDetails().error(description).user((String)null).assertEvent();
        }
        finally {
            this.revertJwksUriSettings(clientRepresentation, clientResource);
        }
    }

    @Test
    public void testDirectGrantRequestFailureES256() throws Exception {
        this.testDirectGrantRequestFailure("ES256");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void testDirectGrantRequestFailure(String algorithm) throws Exception {
        ClientRepresentation clientRepresentation = app2;
        ClientResource clientResource = this.getClient(testRealm.getRealm(), clientRepresentation.getId());
        clientRepresentation = clientResource.toRepresentation();
        try {
            this.setupJwksUrl(algorithm, clientRepresentation, clientResource);
            this.oauth.clientId("client2");
            OAuthClient.AccessTokenResponse response = this.doGrantAccessTokenRequest("test-user@localhost", "password", this.getClient2SignedJWT());
            org.junit.Assert.assertEquals((long)400L, (long)response.getStatusCode());
            org.junit.Assert.assertEquals((Object)"invalid_client", (Object)response.getError());
            this.events.expect(EventType.LOGIN_ERROR).client("client2").session((String)null).clearDetails().error("client_credentials_setup_required").user((String)null).assertEvent();
        }
        finally {
            this.revertJwksUriSettings(clientRepresentation, clientResource);
        }
    }

    private OAuthClient.AccessTokenResponse doAccessTokenRequest(String code, String signedJwt) throws Exception {
        LinkedList<NameValuePair> parameters = new LinkedList<NameValuePair>();
        parameters.add((NameValuePair)new BasicNameValuePair("grant_type", "authorization_code"));
        parameters.add((NameValuePair)new BasicNameValuePair("code", code));
        parameters.add((NameValuePair)new BasicNameValuePair("redirect_uri", this.oauth.getRedirectUri()));
        parameters.add((NameValuePair)new BasicNameValuePair("client_assertion_type", "urn:ietf:params:oauth:client-assertion-type:jwt-bearer"));
        parameters.add((NameValuePair)new BasicNameValuePair("client_assertion", signedJwt));
        CloseableHttpResponse response = this.sendRequest(this.oauth.getAccessTokenUrl(), parameters);
        return new OAuthClient.AccessTokenResponse(response);
    }

    private OAuthClient.AccessTokenResponse doRefreshTokenRequest(String refreshToken, String signedJwt) throws Exception {
        LinkedList<NameValuePair> parameters = new LinkedList<NameValuePair>();
        parameters.add((NameValuePair)new BasicNameValuePair("grant_type", "refresh_token"));
        parameters.add((NameValuePair)new BasicNameValuePair("refresh_token", refreshToken));
        parameters.add((NameValuePair)new BasicNameValuePair("client_assertion_type", "urn:ietf:params:oauth:client-assertion-type:jwt-bearer"));
        parameters.add((NameValuePair)new BasicNameValuePair("client_assertion", signedJwt));
        CloseableHttpResponse response = this.sendRequest(this.oauth.getRefreshTokenUrl(), parameters);
        return new OAuthClient.AccessTokenResponse(response);
    }

    private HttpResponse doLogout(String refreshToken, String signedJwt) throws Exception {
        LinkedList<NameValuePair> parameters = new LinkedList<NameValuePair>();
        parameters.add((NameValuePair)new BasicNameValuePair("grant_type", "refresh_token"));
        parameters.add((NameValuePair)new BasicNameValuePair("refresh_token", refreshToken));
        parameters.add((NameValuePair)new BasicNameValuePair("client_assertion_type", "urn:ietf:params:oauth:client-assertion-type:jwt-bearer"));
        parameters.add((NameValuePair)new BasicNameValuePair("client_assertion", signedJwt));
        return this.sendRequest(this.oauth.getLogoutUrl().build(), parameters);
    }

    private OAuthClient.AccessTokenResponse doClientCredentialsGrantRequest(String signedJwt) throws Exception {
        LinkedList<NameValuePair> parameters = new LinkedList<NameValuePair>();
        parameters.add((NameValuePair)new BasicNameValuePair("grant_type", "client_credentials"));
        parameters.add((NameValuePair)new BasicNameValuePair("client_assertion_type", "urn:ietf:params:oauth:client-assertion-type:jwt-bearer"));
        parameters.add((NameValuePair)new BasicNameValuePair("client_assertion", signedJwt));
        CloseableHttpResponse response = this.sendRequest(this.oauth.getServiceAccountUrl(), parameters);
        return new OAuthClient.AccessTokenResponse(response);
    }

    private OAuthClient.AccessTokenResponse doGrantAccessTokenRequest(String username, String password, String signedJwt) throws Exception {
        LinkedList<NameValuePair> parameters = new LinkedList<NameValuePair>();
        parameters.add((NameValuePair)new BasicNameValuePair("grant_type", "password"));
        parameters.add((NameValuePair)new BasicNameValuePair("username", username));
        parameters.add((NameValuePair)new BasicNameValuePair("password", password));
        parameters.add((NameValuePair)new BasicNameValuePair("client_assertion_type", "urn:ietf:params:oauth:client-assertion-type:jwt-bearer"));
        parameters.add((NameValuePair)new BasicNameValuePair("client_assertion", signedJwt));
        CloseableHttpResponse response = this.sendRequest(this.oauth.getResourceOwnerPasswordCredentialGrantUrl(), parameters);
        return new OAuthClient.AccessTokenResponse(response);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private CloseableHttpResponse sendRequest(String requestUrl, List<NameValuePair> parameters) throws Exception {
        DefaultHttpClient client = new DefaultHttpClient();
        try {
            HttpPost post = new HttpPost(requestUrl);
            UrlEncodedFormEntity formEntity = new UrlEncodedFormEntity(parameters, "UTF-8");
            post.setEntity((HttpEntity)formEntity);
            CloseableHttpResponse closeableHttpResponse = client.execute((HttpUriRequest)post);
            return closeableHttpResponse;
        }
        finally {
            this.oauth.closeClient((CloseableHttpClient)client);
        }
    }

    private String getClient1SignedJWT() {
        return this.getClientSignedJWT(this.getClient1KeyPair(), "client1");
    }

    private String getClient2SignedJWT() {
        return this.getClientSignedJWT(this.getClient2KeyPair(), "client2");
    }

    private KeyPair getClient1KeyPair() {
        return KeystoreUtil.loadKeyPairFromKeystore((String)"classpath:client-auth-test/keystore-client1.jks", (String)"storepass", (String)"keypass", (String)"clientkey", (KeystoreUtil.KeystoreFormat)KeystoreUtil.KeystoreFormat.JKS);
    }

    private KeyPair getClient2KeyPair() {
        return KeystoreUtil.loadKeyPairFromKeystore((String)"classpath:client-auth-test/keystore-client2.jks", (String)"storepass", (String)"keypass", (String)"clientkey", (KeystoreUtil.KeystoreFormat)KeystoreUtil.KeystoreFormat.JKS);
    }

    private String getClientSignedJWT(KeyPair keyPair, String clientId) {
        JWTClientCredentialsProvider jwtProvider = new JWTClientCredentialsProvider();
        jwtProvider.setupKeyPair(keyPair);
        jwtProvider.setTokenTimeout(10);
        return jwtProvider.createSignedRequestToken(clientId, this.getRealmInfoUrl());
    }

    private String getRealmInfoUrl() {
        String authServerBaseUrl = UriUtils.getOrigin((String)this.oauth.getRedirectUri()) + "/auth";
        return KeycloakUriBuilder.fromUri((String)authServerBaseUrl).path("/realms/{realm-name}").build(new Object[]{"test"}).toString();
    }

    private ClientAttributeCertificateResource getClientAttributeCertificateResource(String realm, String clientId) {
        return this.getClient(realm, clientId).getCertficateResource("jwt.credential");
    }

    private ClientResource getClient(String realm, String clientId) {
        return this.realmsResouce().realm(realm).clients().get(clientId);
    }

    private static KeyStore getKeystore(InputStream is, String storePassword, String format) throws Exception {
        KeyStore keyStore = format.equals("JKS") ? KeyStore.getInstance(format) : KeyStore.getInstance(format, "BC");
        keyStore.load(is, storePassword.toCharArray());
        return keyStore;
    }

    private KeyPair setupJwksUrl(String algorithm, ClientRepresentation clientRepresentation, ClientResource clientResource) throws Exception {
        return this.setupJwksUrl(algorithm, true, clientRepresentation, clientResource);
    }

    private KeyPair setupJwksUrl(String algorithm, boolean advertiseJWKAlgorithm, ClientRepresentation clientRepresentation, ClientResource clientResource) throws Exception {
        TestOIDCEndpointsApplicationResource oidcClientEndpointsResource = this.testingClient.testApp().oidcClientEndpoints();
        oidcClientEndpointsResource.generateKeys(algorithm, Boolean.valueOf(advertiseJWKAlgorithm));
        Map generatedKeys = oidcClientEndpointsResource.getKeysAsBase64();
        KeyPair keyPair = this.getKeyPairFromGeneratedBase64(generatedKeys, algorithm);
        OIDCAdvancedConfigWrapper.fromClientRepresentation((ClientRepresentation)clientRepresentation).setUseJwksUrl(true);
        String jwksUrl = TestApplicationResourceUrls.clientJwksUri();
        OIDCAdvancedConfigWrapper.fromClientRepresentation((ClientRepresentation)clientRepresentation).setJwksUrl(jwksUrl);
        clientResource.update(clientRepresentation);
        this.setTimeOffset(20);
        return keyPair;
    }

    private KeyPair setupJwks(String algorithm, ClientRepresentation clientRepresentation, ClientResource clientResource) throws Exception {
        TestOIDCEndpointsApplicationResource oidcClientEndpointsResource = this.testingClient.testApp().oidcClientEndpoints();
        oidcClientEndpointsResource.generateKeys(algorithm);
        Map generatedKeys = oidcClientEndpointsResource.getKeysAsBase64();
        KeyPair keyPair = this.getKeyPairFromGeneratedBase64(generatedKeys, algorithm);
        OIDCAdvancedConfigWrapper.fromClientRepresentation((ClientRepresentation)clientRepresentation).setUseJwksString(true);
        JSONWebKeySet keySet = oidcClientEndpointsResource.getJwks();
        OIDCAdvancedConfigWrapper.fromClientRepresentation((ClientRepresentation)clientRepresentation).setJwksString(JsonSerialization.writeValueAsString((Object)keySet));
        clientResource.update(clientRepresentation);
        this.setTimeOffset(20);
        return keyPair;
    }

    private void revertJwksUriSettings(ClientRepresentation clientRepresentation, ClientResource clientResource) {
        OIDCAdvancedConfigWrapper.fromClientRepresentation((ClientRepresentation)clientRepresentation).setUseJwksUrl(false);
        OIDCAdvancedConfigWrapper.fromClientRepresentation((ClientRepresentation)clientRepresentation).setJwksUrl(null);
        clientResource.update(clientRepresentation);
    }

    private void revertJwksSettings(ClientRepresentation clientRepresentation, ClientResource clientResource) {
        OIDCAdvancedConfigWrapper.fromClientRepresentation((ClientRepresentation)clientRepresentation).setUseJwksString(false);
        OIDCAdvancedConfigWrapper.fromClientRepresentation((ClientRepresentation)clientRepresentation).setJwksString(null);
        clientResource.update(clientRepresentation);
    }

    private KeyPair getKeyPairFromGeneratedBase64(Map<String, String> generatedKeys, String algorithm) throws Exception {
        String privateKeyBase64 = generatedKeys.get("privateKey");
        String publicKeyBase64 = generatedKeys.get("publicKey");
        PrivateKey privateKey = ClientAuthSignedJWTTest.decodePrivateKey(Base64.decode((String)privateKeyBase64), algorithm);
        PublicKey publicKey = ClientAuthSignedJWTTest.decodePublicKey(Base64.decode((String)publicKeyBase64), algorithm);
        return new KeyPair(publicKey, privateKey);
    }

    private static PrivateKey decodePrivateKey(byte[] der, String algorithm) throws NoSuchAlgorithmException, InvalidKeySpecException, NoSuchProviderException {
        PKCS8EncodedKeySpec spec = new PKCS8EncodedKeySpec(der);
        String keyAlg = ClientAuthSignedJWTTest.getKeyAlgorithmFromJwaAlgorithm(algorithm);
        KeyFactory kf = KeyFactory.getInstance(keyAlg, "BC");
        return kf.generatePrivate(spec);
    }

    private static PublicKey decodePublicKey(byte[] der, String algorithm) throws NoSuchAlgorithmException, InvalidKeySpecException, NoSuchProviderException {
        X509EncodedKeySpec spec = new X509EncodedKeySpec(der);
        String keyAlg = ClientAuthSignedJWTTest.getKeyAlgorithmFromJwaAlgorithm(algorithm);
        KeyFactory kf = KeyFactory.getInstance(keyAlg, "BC");
        return kf.generatePublic(spec);
    }

    private String createSignedRequestToken(String clientId, String realmInfoUrl, PrivateKey privateKey, PublicKey publicKey, String algorithm) {
        return this.createSignledRequestToken(privateKey, publicKey, algorithm, this.createRequestToken(clientId, realmInfoUrl));
    }

    private String createSignledRequestToken(PrivateKey privateKey, PublicKey publicKey, String algorithm, JsonWebToken jwt) {
        String kid = KeyUtils.createKeyId((Key)publicKey);
        SignatureSignerContext signer = this.oauth.createSigner(privateKey, kid, algorithm);
        String ret = new JWSBuilder().kid(kid).jsonContent((Object)jwt).sign(signer);
        return ret;
    }

    private JsonWebToken createRequestToken(String clientId, String realmInfoUrl) {
        JsonWebToken reqToken = new JsonWebToken();
        reqToken.id(AdapterUtils.generateId());
        reqToken.issuer(clientId);
        reqToken.subject(clientId);
        reqToken.audience(new String[]{realmInfoUrl});
        int now = Time.currentTime();
        reqToken.issuedAt(now);
        reqToken.expiration(now + 10);
        reqToken.notBefore(now);
        return reqToken;
    }

    private static String getKeyAlgorithmFromJwaAlgorithm(String jwaAlgorithm) {
        String keyAlg = null;
        switch (jwaAlgorithm) {
            case "RS256": 
            case "RS384": 
            case "RS512": 
            case "PS256": 
            case "PS384": 
            case "PS512": {
                keyAlg = "RSA";
                break;
            }
            case "ES256": 
            case "ES384": 
            case "ES512": {
                keyAlg = "EC";
                break;
            }
            default: {
                throw new RuntimeException("Unsupported signature algorithm");
            }
        }
        return keyAlg;
    }

    protected class CustomJWTClientCredentialsProvider
    extends JWTClientCredentialsProvider {
        private Map<String, Boolean> enabledClaims = new HashMap<String, Boolean>();

        public CustomJWTClientCredentialsProvider() {
            String[] claims;
            for (String claim : claims = new String[]{"id", "issuer", "subject", "audience", "expiration", "notBefore", "issuedAt"}) {
                this.enabledClaims.put(claim, true);
            }
        }

        public void enableClaim(String claim, boolean value) {
            if (!this.enabledClaims.containsKey(claim)) {
                throw new IllegalArgumentException("Claim \"" + claim + "\" doesn't exist");
            }
            this.enabledClaims.put(claim, value);
        }

        public boolean isClaimEnabled(String claim) {
            Boolean value = this.enabledClaims.get(claim);
            if (value == null) {
                throw new IllegalArgumentException("Claim \"" + claim + "\" doesn't exist");
            }
            return value;
        }

        public Set<String> getClaims() {
            return this.enabledClaims.keySet();
        }

        protected JsonWebToken createRequestToken(String clientId, String realmInfoUrl) {
            JsonWebToken reqToken = new JsonWebToken();
            if (this.isClaimEnabled("id")) {
                reqToken.id(AdapterUtils.generateId());
            }
            if (this.isClaimEnabled("issuer")) {
                reqToken.issuer(clientId);
            }
            if (this.isClaimEnabled("subject")) {
                reqToken.subject(clientId);
            }
            if (this.isClaimEnabled("audience")) {
                reqToken.audience(new String[]{realmInfoUrl});
            }
            int now = Time.currentTime();
            if (this.isClaimEnabled("issuedAt")) {
                reqToken.issuedAt(now);
            }
            if (this.isClaimEnabled("expiration")) {
                reqToken.expiration(now + this.getTokenTimeout());
            }
            if (this.isClaimEnabled("notBefore")) {
                reqToken.notBefore(now);
            }
            return reqToken;
        }
    }
}

