/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.testsuite.admin;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import javax.ws.rs.ClientErrorException;
import javax.ws.rs.NotFoundException;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import org.hamcrest.Matcher;
import org.hamcrest.Matchers;
import org.jboss.resteasy.plugins.providers.multipart.MultipartFormDataOutput;
import org.junit.Test;
import org.keycloak.admin.client.resource.IdentityProviderResource;
import org.keycloak.common.enums.SslRequired;
import org.keycloak.dom.saml.v2.metadata.EndpointType;
import org.keycloak.dom.saml.v2.metadata.EntityDescriptorType;
import org.keycloak.dom.saml.v2.metadata.IndexedEndpointType;
import org.keycloak.dom.saml.v2.metadata.KeyDescriptorType;
import org.keycloak.dom.saml.v2.metadata.KeyTypes;
import org.keycloak.dom.saml.v2.metadata.SPSSODescriptorType;
import org.keycloak.events.admin.OperationType;
import org.keycloak.events.admin.ResourceType;
import org.keycloak.models.IdentityProviderMapperSyncMode;
import org.keycloak.models.utils.StripSecretsUtils;
import org.keycloak.representations.idm.AdminEventRepresentation;
import org.keycloak.representations.idm.ErrorRepresentation;
import org.keycloak.representations.idm.IdentityProviderMapperRepresentation;
import org.keycloak.representations.idm.IdentityProviderMapperTypeRepresentation;
import org.keycloak.representations.idm.IdentityProviderRepresentation;
import org.keycloak.saml.common.constants.JBossSAMLURIConstants;
import org.keycloak.saml.common.exceptions.ConfigurationException;
import org.keycloak.saml.common.exceptions.ParsingException;
import org.keycloak.saml.common.exceptions.ProcessingException;
import org.keycloak.saml.common.util.DocumentUtil;
import org.keycloak.saml.processing.core.parsers.saml.SAMLParser;
import org.keycloak.testsuite.Assert;
import org.keycloak.testsuite.admin.AbstractAdminTest;
import org.keycloak.testsuite.admin.ApiUtil;
import org.keycloak.testsuite.arquillian.annotation.AuthServerContainerExclude;
import org.keycloak.testsuite.broker.OIDCIdentityProviderConfigRep;
import org.keycloak.testsuite.updaters.RealmAttributeUpdater;
import org.keycloak.testsuite.updaters.ServerResourceUpdater;
import org.keycloak.testsuite.util.AdminEventPaths;
import org.keycloak.testsuite.util.OAuthClient;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

public class IdentityProviderTest
extends AbstractAdminTest {
    private static final String SIGNING_CERT_1 = "MIICmzCCAYMCBgFUYnC0OjANBgkqhkiG9w0BAQsFADARMQ8wDQYDVQQDDAZtYXN0ZXIwHhcNMTYwNDI5MTQzMjEzWhcNMjYwNDI5MTQzMzUzWjARMQ8wDQYDVQQDDAZtYXN0ZXIwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCN25AW1poMEZRbuMAHG58AThZmCwMV6/Gcui4mjGacRFyudgqzLjQ2rxpoW41JAtLjbjeAhuWvirUcFVcOeS3gM/ZC27qCpYighAcylZz6MYocnEe1+e8rPPk4JlID6Wv62dgu+pL/vYsQpRhvD3Y2c/ytgr5D32xF+KnzDehUy5BSyzypvu12Wq9mS5vK5tzkN37EjkhpY2ZxaXPubjDIITCAL4Q8M/m5IlacBaUZbzI4AQrHnMP1O1IH2dHSWuMiBe+xSDTco72PmuYPJKTV4wQdeBUIkYbfLc4RxVmXEvgkQgyW86EoMPxlWJpj7+mTIR+l+2thZPr/VgwTs82rAgMBAAEwDQYJKoZIhvcNAQELBQADggEBAA/Ip/Hi8RoVu5ouaFFlc5whT7ltuK8slfLGW4tM4vJXhInYwsqIRQKBNDYW/64xle3eII4u1yAH1OYRRwEs7Em1pr4QuFuTY1at+aE0sE46XDlyESI0txJjWxYoT133vM0We2pj1b2nxgU30rwjKA3whnKEfTEYT/n3JBSqNggy6l8ZGw/oPSgvPaR4+xeB1tfQFC4VrLoYKoqH6hAL530nKxL+qV8AIfL64NDEE8ankIAEDAAFe8x3CPUfXR/p4KOANKkpz8ieQaHDb1eITkAwUwjESj6UF9D1aePlhWls/HX0gujFXtWfWfrJ8CU/ogwlH8y1jgRuLjFQYZk6llc=";
    private static final String SIGNING_CERT_2 = "MIIBnDCCAQUCBgFYKXKsPTANBgkqhkiG9w0BAQsFADAUMRIwEAYDVQQDDAlzYW1sLWRlbW8wHhcNMTYxMTAzMDkwNzEwWhcNMjYxMTAzMDkwODUwWjAUMRIwEAYDVQQDDAlzYW1sLWRlbW8wgZ8wDQYJKoZIhvcNAQEBBQADgY0AMIGJAoGBAKtWsK5O0CtuBpnMvWG+HTG0vmZzujQ2o9WdheQu+BzCILcGMsbDW0YQaglpcO5JpGWWhubnckGGPHfdQ2/7nP9QwbiTK0FbGF41UqcvoaCqU1psxoV88s8IXyQCAqeyLv00yj6foqdJjxh5SZ5z+na+M7Y2OxIBVxYRAxWEnfUvAgMBAAEwDQYJKoZIhvcNAQELBQADgYEAhetvOU8TyqfZF5jpv0IcrviLl/DoFrbjByeHR+pu/vClcAOjL/u7oQELuuTfNsBI4tpexUj5G8q/YbEz0gk7idfLXrAUVcsR73oTngrhRfwUSmPrjjK0kjcRb6HL9V/+wh3R/6mEd59U08ExT8N38rhmn0CI3ehMdebReprP7U8=";

    @Test
    public void testFindAll() {
        this.create(this.createRep("google", "google"));
        this.create(this.createRep("facebook", "facebook"));
        Assert.assertNames(this.realm.identityProviders().findAll(), "google", "facebook");
    }

    @Test
    public void testCreateWithReservedCharacterForAlias() {
        IdentityProviderRepresentation newIdentityProvider = this.createRep("ne$&w-identity-provider", "oidc");
        newIdentityProvider.getConfig().put("clientId", "clientId");
        newIdentityProvider.getConfig().put("clientSecret", "some secret value");
        Response response = this.realm.identityProviders().create(newIdentityProvider);
        Assert.assertEquals((long)400L, (long)response.getStatus());
    }

    @Test
    public void testCreate() {
        IdentityProviderRepresentation newIdentityProvider = this.createRep("new-identity-provider", "oidc");
        newIdentityProvider.getConfig().put("syncMode", "IMPORT");
        newIdentityProvider.getConfig().put("clientId", "clientId");
        newIdentityProvider.getConfig().put("clientSecret", "some secret value");
        this.create(newIdentityProvider);
        IdentityProviderResource identityProviderResource = this.realm.identityProviders().get("new-identity-provider");
        org.junit.Assert.assertNotNull((Object)identityProviderResource);
        IdentityProviderRepresentation representation = identityProviderResource.toRepresentation();
        org.junit.Assert.assertNotNull((Object)representation);
        org.junit.Assert.assertNotNull((Object)representation.getInternalId());
        org.junit.Assert.assertEquals((Object)"new-identity-provider", (Object)representation.getAlias());
        org.junit.Assert.assertEquals((Object)"oidc", (Object)representation.getProviderId());
        org.junit.Assert.assertEquals((Object)"IMPORT", representation.getConfig().get("syncMode"));
        org.junit.Assert.assertEquals((Object)"clientId", representation.getConfig().get("clientId"));
        org.junit.Assert.assertEquals((Object)"**********", representation.getConfig().get("clientSecret"));
        org.junit.Assert.assertTrue((boolean)representation.isEnabled());
        org.junit.Assert.assertFalse((boolean)representation.isStoreToken());
        org.junit.Assert.assertFalse((boolean)representation.isTrustEmail());
        org.junit.Assert.assertEquals((Object)"some secret value", this.testingClient.testing("admin-client-test").getIdentityProviderConfig("new-identity-provider").get("clientSecret"));
        IdentityProviderRepresentation rep = this.realm.identityProviders().findAll().stream().filter(i -> i.getAlias().equals("new-identity-provider")).findFirst().get();
        org.junit.Assert.assertEquals((Object)"**********", rep.getConfig().get("clientSecret"));
    }

    @Test
    @AuthServerContainerExclude(value={AuthServerContainerExclude.AuthServer.REMOTE})
    public void failCreateInvalidUrl() throws Exception {
        try (ServerResourceUpdater c = ((RealmAttributeUpdater)new RealmAttributeUpdater(this.realmsResouce().realm("test")).updateWith(r -> r.setSslRequired(SslRequired.ALL.name()))).update();){
            ErrorRepresentation error;
            IdentityProviderRepresentation newIdentityProvider = this.createRep("new-identity-provider", "oidc");
            newIdentityProvider.getConfig().put("clientId", "clientId");
            newIdentityProvider.getConfig().put("clientSecret", "some secret value");
            OIDCIdentityProviderConfigRep oidcConfig = new OIDCIdentityProviderConfigRep(newIdentityProvider);
            oidcConfig.setAuthorizationUrl("invalid://test");
            try (Response response = this.realm.identityProviders().create(newIdentityProvider);){
                org.junit.Assert.assertEquals((long)Response.Status.BAD_REQUEST.getStatusCode(), (long)response.getStatus());
                error = (ErrorRepresentation)response.readEntity(ErrorRepresentation.class);
                org.junit.Assert.assertEquals((Object)"The url [authorization_url] is malformed", (Object)error.getErrorMessage());
            }
            oidcConfig.setAuthorizationUrl(null);
            oidcConfig.setTokenUrl("http://test");
            response = this.realm.identityProviders().create(newIdentityProvider);
            try {
                org.junit.Assert.assertEquals((long)Response.Status.BAD_REQUEST.getStatusCode(), (long)response.getStatus());
                error = (ErrorRepresentation)response.readEntity(ErrorRepresentation.class);
                org.junit.Assert.assertEquals((Object)"The url [token_url] requires secure connections", (Object)error.getErrorMessage());
            }
            finally {
                if (response != null) {
                    response.close();
                }
            }
            oidcConfig.setAuthorizationUrl(null);
            oidcConfig.setTokenUrl(null);
            oidcConfig.setJwksUrl("http://test");
            response = this.realm.identityProviders().create(newIdentityProvider);
            try {
                org.junit.Assert.assertEquals((long)Response.Status.BAD_REQUEST.getStatusCode(), (long)response.getStatus());
                error = (ErrorRepresentation)response.readEntity(ErrorRepresentation.class);
                org.junit.Assert.assertEquals((Object)"The url [jwks_url] requires secure connections", (Object)error.getErrorMessage());
            }
            finally {
                if (response != null) {
                    response.close();
                }
            }
            oidcConfig.setAuthorizationUrl(null);
            oidcConfig.setTokenUrl(null);
            oidcConfig.setJwksUrl(null);
            oidcConfig.setLogoutUrl("http://test");
            response = this.realm.identityProviders().create(newIdentityProvider);
            try {
                org.junit.Assert.assertEquals((long)Response.Status.BAD_REQUEST.getStatusCode(), (long)response.getStatus());
                error = (ErrorRepresentation)response.readEntity(ErrorRepresentation.class);
                org.junit.Assert.assertEquals((Object)"The url [logout_url] requires secure connections", (Object)error.getErrorMessage());
            }
            finally {
                if (response != null) {
                    response.close();
                }
            }
            oidcConfig.setAuthorizationUrl(null);
            oidcConfig.setTokenUrl(null);
            oidcConfig.setJwksUrl(null);
            oidcConfig.setLogoutUrl(null);
            oidcConfig.setUserInfoUrl("http://test");
            response = this.realm.identityProviders().create(newIdentityProvider);
            try {
                org.junit.Assert.assertEquals((long)Response.Status.BAD_REQUEST.getStatusCode(), (long)response.getStatus());
                error = (ErrorRepresentation)response.readEntity(ErrorRepresentation.class);
                org.junit.Assert.assertEquals((Object)"The url [userinfo_url] requires secure connections", (Object)error.getErrorMessage());
            }
            finally {
                if (response != null) {
                    response.close();
                }
            }
        }
    }

    @Test
    public void testCreateWithBasicAuth() {
        IdentityProviderRepresentation newIdentityProvider = this.createRep("new-identity-provider", "oidc");
        newIdentityProvider.getConfig().put("syncMode", "IMPORT");
        newIdentityProvider.getConfig().put("clientId", "clientId");
        newIdentityProvider.getConfig().put("clientSecret", "some secret value");
        newIdentityProvider.getConfig().put("clientAuthMethod", "client_secret_basic");
        this.create(newIdentityProvider);
        IdentityProviderResource identityProviderResource = this.realm.identityProviders().get("new-identity-provider");
        org.junit.Assert.assertNotNull((Object)identityProviderResource);
        IdentityProviderRepresentation representation = identityProviderResource.toRepresentation();
        org.junit.Assert.assertNotNull((Object)representation);
        org.junit.Assert.assertNotNull((Object)representation.getInternalId());
        org.junit.Assert.assertEquals((Object)"new-identity-provider", (Object)representation.getAlias());
        org.junit.Assert.assertEquals((Object)"oidc", (Object)representation.getProviderId());
        org.junit.Assert.assertEquals((Object)"IMPORT", representation.getConfig().get("syncMode"));
        org.junit.Assert.assertEquals((Object)"clientId", representation.getConfig().get("clientId"));
        org.junit.Assert.assertEquals((Object)"**********", representation.getConfig().get("clientSecret"));
        org.junit.Assert.assertEquals((Object)"client_secret_basic", representation.getConfig().get("clientAuthMethod"));
        org.junit.Assert.assertTrue((boolean)representation.isEnabled());
        org.junit.Assert.assertFalse((boolean)representation.isStoreToken());
        org.junit.Assert.assertFalse((boolean)representation.isTrustEmail());
        org.junit.Assert.assertEquals((Object)"some secret value", this.testingClient.testing("admin-client-test").getIdentityProviderConfig("new-identity-provider").get("clientSecret"));
        IdentityProviderRepresentation rep = this.realm.identityProviders().findAll().stream().filter(i -> i.getAlias().equals("new-identity-provider")).findFirst().get();
        org.junit.Assert.assertEquals((Object)"**********", rep.getConfig().get("clientSecret"));
    }

    @Test
    public void testCreateWithJWT() {
        IdentityProviderRepresentation newIdentityProvider = this.createRep("new-identity-provider", "oidc");
        newIdentityProvider.getConfig().put("syncMode", "IMPORT");
        newIdentityProvider.getConfig().put("clientId", "clientId");
        newIdentityProvider.getConfig().put("clientAuthMethod", "private_key_jwt");
        this.create(newIdentityProvider);
        IdentityProviderResource identityProviderResource = this.realm.identityProviders().get("new-identity-provider");
        org.junit.Assert.assertNotNull((Object)identityProviderResource);
        IdentityProviderRepresentation representation = identityProviderResource.toRepresentation();
        org.junit.Assert.assertNotNull((Object)representation);
        org.junit.Assert.assertNotNull((Object)representation.getInternalId());
        org.junit.Assert.assertEquals((Object)"new-identity-provider", (Object)representation.getAlias());
        org.junit.Assert.assertEquals((Object)"oidc", (Object)representation.getProviderId());
        org.junit.Assert.assertEquals((Object)"IMPORT", representation.getConfig().get("syncMode"));
        org.junit.Assert.assertEquals((Object)"clientId", representation.getConfig().get("clientId"));
        org.junit.Assert.assertNull(representation.getConfig().get("clientSecret"));
        org.junit.Assert.assertEquals((Object)"private_key_jwt", representation.getConfig().get("clientAuthMethod"));
        org.junit.Assert.assertTrue((boolean)representation.isEnabled());
        org.junit.Assert.assertFalse((boolean)representation.isStoreToken());
        org.junit.Assert.assertFalse((boolean)representation.isTrustEmail());
    }

    @Test
    public void testUpdate() {
        IdentityProviderRepresentation newIdentityProvider = this.createRep("update-identity-provider", "oidc");
        newIdentityProvider.getConfig().put("syncMode", "IMPORT");
        newIdentityProvider.getConfig().put("clientId", "clientId");
        newIdentityProvider.getConfig().put("clientSecret", "some secret value");
        this.create(newIdentityProvider);
        IdentityProviderResource identityProviderResource = this.realm.identityProviders().get("update-identity-provider");
        org.junit.Assert.assertNotNull((Object)identityProviderResource);
        IdentityProviderRepresentation representation = identityProviderResource.toRepresentation();
        org.junit.Assert.assertNotNull((Object)representation);
        org.junit.Assert.assertEquals((Object)"update-identity-provider", (Object)representation.getAlias());
        representation.setAlias("changed-alias");
        representation.setEnabled(false);
        representation.setStoreToken(true);
        representation.getConfig().put("clientId", "changedClientId");
        identityProviderResource.update(representation);
        AdminEventRepresentation event = this.assertAdminEvents.assertEvent(this.realmId, OperationType.UPDATE, AdminEventPaths.identityProviderPath("update-identity-provider"), (Object)representation, ResourceType.IDENTITY_PROVIDER);
        org.junit.Assert.assertFalse((boolean)event.getRepresentation().contains("some secret value"));
        org.junit.Assert.assertTrue((boolean)event.getRepresentation().contains("**********"));
        identityProviderResource = this.realm.identityProviders().get(representation.getInternalId());
        org.junit.Assert.assertNotNull((Object)identityProviderResource);
        representation = identityProviderResource.toRepresentation();
        org.junit.Assert.assertFalse((boolean)representation.isEnabled());
        org.junit.Assert.assertTrue((boolean)representation.isStoreToken());
        org.junit.Assert.assertEquals((Object)"changedClientId", representation.getConfig().get("clientId"));
        org.junit.Assert.assertEquals((Object)"some secret value", this.testingClient.testing("admin-client-test").getIdentityProviderConfig("changed-alias").get("clientSecret"));
        representation.getConfig().put("clientSecret", "${vault.key}");
        identityProviderResource.update(representation);
        event = this.assertAdminEvents.assertEvent(this.realmId, OperationType.UPDATE, AdminEventPaths.identityProviderPath(representation.getInternalId()), (Object)representation, ResourceType.IDENTITY_PROVIDER);
        org.junit.Assert.assertThat((Object)event.getRepresentation(), (Matcher)Matchers.containsString((String)"${vault.key}"));
        org.junit.Assert.assertThat((Object)event.getRepresentation(), (Matcher)Matchers.not((Matcher)Matchers.containsString((String)"**********")));
        org.junit.Assert.assertThat((Object)identityProviderResource.toRepresentation().getConfig(), (Matcher)Matchers.hasEntry((Object)"clientSecret", (Object)"${vault.key}"));
        org.junit.Assert.assertEquals((Object)"${vault.key}", this.testingClient.testing("admin-client-test").getIdentityProviderConfig("changed-alias").get("clientSecret"));
    }

    @Test
    public void failUpdateInvalidUrl() throws Exception {
        try (RealmAttributeUpdater rau = (RealmAttributeUpdater)((RealmAttributeUpdater)new RealmAttributeUpdater(this.realm).updateWith(r -> r.setSslRequired(SslRequired.ALL.name()))).update();){
            ErrorRepresentation error;
            Response response;
            IdentityProviderRepresentation representation = this.createRep(UUID.randomUUID().toString(), "oidc");
            representation.getConfig().put("clientId", "clientId");
            representation.getConfig().put("clientSecret", "some secret value");
            try (Response response2 = this.realm.identityProviders().create(representation);){
                org.junit.Assert.assertEquals((long)Response.Status.CREATED.getStatusCode(), (long)response2.getStatus());
            }
            IdentityProviderResource resource = this.realm.identityProviders().get(representation.getAlias());
            representation = resource.toRepresentation();
            OIDCIdentityProviderConfigRep oidcConfig = new OIDCIdentityProviderConfigRep(representation);
            oidcConfig.setAuthorizationUrl("invalid://test");
            try {
                resource.update(representation);
                org.junit.Assert.fail((String)"Invalid URL");
            }
            catch (Exception e) {
                org.junit.Assert.assertTrue((boolean)(e instanceof ClientErrorException));
                response = ((ClientErrorException)((Object)ClientErrorException.class.cast(e))).getResponse();
                org.junit.Assert.assertEquals((long)Response.Status.BAD_REQUEST.getStatusCode(), (long)response.getStatus());
                error = (ErrorRepresentation)((ClientErrorException)((Object)e)).getResponse().readEntity(ErrorRepresentation.class);
                org.junit.Assert.assertEquals((Object)"The url [authorization_url] is malformed", (Object)error.getErrorMessage());
            }
            oidcConfig.setAuthorizationUrl(null);
            oidcConfig.setTokenUrl("http://test");
            try {
                resource.update(representation);
                org.junit.Assert.fail((String)"Invalid URL");
            }
            catch (Exception e) {
                org.junit.Assert.assertTrue((boolean)(e instanceof ClientErrorException));
                response = ((ClientErrorException)((Object)ClientErrorException.class.cast(e))).getResponse();
                org.junit.Assert.assertEquals((long)Response.Status.BAD_REQUEST.getStatusCode(), (long)response.getStatus());
                error = (ErrorRepresentation)((ClientErrorException)((Object)e)).getResponse().readEntity(ErrorRepresentation.class);
                org.junit.Assert.assertEquals((Object)"The url [token_url] requires secure connections", (Object)error.getErrorMessage());
            }
            oidcConfig.setAuthorizationUrl(null);
            oidcConfig.setTokenUrl(null);
            oidcConfig.setJwksUrl("http://test");
            try {
                resource.update(representation);
                org.junit.Assert.fail((String)"Invalid URL");
            }
            catch (Exception e) {
                org.junit.Assert.assertTrue((boolean)(e instanceof ClientErrorException));
                response = ((ClientErrorException)((Object)ClientErrorException.class.cast(e))).getResponse();
                org.junit.Assert.assertEquals((long)Response.Status.BAD_REQUEST.getStatusCode(), (long)response.getStatus());
                error = (ErrorRepresentation)((ClientErrorException)((Object)e)).getResponse().readEntity(ErrorRepresentation.class);
                org.junit.Assert.assertEquals((Object)"The url [jwks_url] requires secure connections", (Object)error.getErrorMessage());
            }
            oidcConfig.setAuthorizationUrl(null);
            oidcConfig.setTokenUrl(null);
            oidcConfig.setJwksUrl(null);
            oidcConfig.setLogoutUrl("http://test");
            try {
                resource.update(representation);
                org.junit.Assert.fail((String)"Invalid URL");
            }
            catch (Exception e) {
                org.junit.Assert.assertTrue((boolean)(e instanceof ClientErrorException));
                response = ((ClientErrorException)((Object)ClientErrorException.class.cast(e))).getResponse();
                org.junit.Assert.assertEquals((long)Response.Status.BAD_REQUEST.getStatusCode(), (long)response.getStatus());
                error = (ErrorRepresentation)((ClientErrorException)((Object)e)).getResponse().readEntity(ErrorRepresentation.class);
                org.junit.Assert.assertEquals((Object)"The url [logout_url] requires secure connections", (Object)error.getErrorMessage());
            }
            oidcConfig.setAuthorizationUrl(null);
            oidcConfig.setTokenUrl(null);
            oidcConfig.setJwksUrl(null);
            oidcConfig.setLogoutUrl(null);
            oidcConfig.setUserInfoUrl("http://localhost");
            try {
                resource.update(representation);
                org.junit.Assert.fail((String)"Invalid URL");
            }
            catch (Exception e) {
                org.junit.Assert.assertTrue((boolean)(e instanceof ClientErrorException));
                response = ((ClientErrorException)((Object)ClientErrorException.class.cast(e))).getResponse();
                org.junit.Assert.assertEquals((long)Response.Status.BAD_REQUEST.getStatusCode(), (long)response.getStatus());
                error = (ErrorRepresentation)((ClientErrorException)((Object)e)).getResponse().readEntity(ErrorRepresentation.class);
                org.junit.Assert.assertEquals((Object)"The url [userinfo_url] requires secure connections", (Object)error.getErrorMessage());
            }
            ((RealmAttributeUpdater)rau.updateWith(r -> r.setSslRequired(SslRequired.EXTERNAL.name()))).update();
            resource.update(representation);
        }
    }

    @Test
    public void testRemove() {
        IdentityProviderRepresentation newIdentityProvider = this.createRep("remove-identity-provider", "saml");
        this.create(newIdentityProvider);
        IdentityProviderResource identityProviderResource = this.realm.identityProviders().get("remove-identity-provider");
        org.junit.Assert.assertNotNull((Object)identityProviderResource);
        IdentityProviderRepresentation representation = identityProviderResource.toRepresentation();
        org.junit.Assert.assertNotNull((Object)representation);
        identityProviderResource.remove();
        this.assertAdminEvents.assertEvent(this.realmId, OperationType.DELETE, AdminEventPaths.identityProviderPath("remove-identity-provider"), ResourceType.IDENTITY_PROVIDER);
        try {
            this.realm.identityProviders().get("remove-identity-provider").toRepresentation();
            Assert.fail((String)"Not expected to found");
        }
        catch (NotFoundException notFoundException) {
            // empty catch block
        }
    }

    private void create(IdentityProviderRepresentation idpRep) {
        Response response = this.realm.identityProviders().create(idpRep);
        Assert.assertNotNull((Object)ApiUtil.getCreatedId((Response)response));
        response.close();
        this.getCleanup().addIdentityProviderAlias(idpRep.getAlias());
        String secret = idpRep.getConfig() != null ? (String)idpRep.getConfig().get("clientSecret") : null;
        idpRep = StripSecretsUtils.strip((IdentityProviderRepresentation)idpRep);
        this.assertAdminEvents.assertEvent(this.realmId, OperationType.CREATE, AdminEventPaths.identityProviderPath(idpRep.getAlias()), (Object)idpRep, ResourceType.IDENTITY_PROVIDER);
        if (secret != null) {
            idpRep.getConfig().put("clientSecret", secret);
        }
    }

    private IdentityProviderRepresentation createRep(String id, String providerId) {
        return this.createRep(id, providerId, true, null);
    }

    private IdentityProviderRepresentation createRep(String id, String providerId, boolean enabled, Map<String, String> config) {
        IdentityProviderRepresentation idp = new IdentityProviderRepresentation();
        idp.setAlias(id);
        idp.setDisplayName(id);
        idp.setProviderId(providerId);
        idp.setEnabled(enabled);
        if (config != null) {
            idp.setConfig(config);
        }
        return idp;
    }

    @Test
    public void testMapperTypes() {
        this.create(this.createRep("google", "google"));
        IdentityProviderResource provider = this.realm.identityProviders().get("google");
        Map mapperTypes = provider.getMapperTypes();
        this.assertMapperTypes(mapperTypes, "google-user-attribute-mapper", "oidc-username-idp-mapper");
        this.create(this.createRep("facebook", "facebook"));
        provider = this.realm.identityProviders().get("facebook");
        mapperTypes = provider.getMapperTypes();
        this.assertMapperTypes(mapperTypes, "facebook-user-attribute-mapper", "oidc-username-idp-mapper");
        this.create(this.createRep("github", "github"));
        provider = this.realm.identityProviders().get("github");
        mapperTypes = provider.getMapperTypes();
        this.assertMapperTypes(mapperTypes, "github-user-attribute-mapper", "oidc-username-idp-mapper");
        this.create(this.createRep("twitter", "twitter"));
        provider = this.realm.identityProviders().get("twitter");
        mapperTypes = provider.getMapperTypes();
        this.assertMapperTypes(mapperTypes, "oidc-username-idp-mapper");
        this.create(this.createRep("linkedin", "linkedin"));
        provider = this.realm.identityProviders().get("linkedin");
        mapperTypes = provider.getMapperTypes();
        this.assertMapperTypes(mapperTypes, "linkedin-user-attribute-mapper", "oidc-username-idp-mapper");
        this.create(this.createRep("microsoft", "microsoft"));
        provider = this.realm.identityProviders().get("microsoft");
        mapperTypes = provider.getMapperTypes();
        this.assertMapperTypes(mapperTypes, "microsoft-user-attribute-mapper", "oidc-username-idp-mapper");
        this.create(this.createRep("stackoverflow", "stackoverflow"));
        provider = this.realm.identityProviders().get("stackoverflow");
        mapperTypes = provider.getMapperTypes();
        this.assertMapperTypes(mapperTypes, "stackoverflow-user-attribute-mapper", "oidc-username-idp-mapper");
        this.create(this.createRep("keycloak-oidc", "keycloak-oidc"));
        provider = this.realm.identityProviders().get("keycloak-oidc");
        mapperTypes = provider.getMapperTypes();
        this.assertMapperTypes(mapperTypes, "keycloak-oidc-role-to-role-idp-mapper", "oidc-user-attribute-idp-mapper", "oidc-role-idp-mapper", "oidc-username-idp-mapper", "oidc-advanced-group-idp-mapper", "oidc-advanced-role-idp-mapper");
        this.create(this.createRep("oidc", "oidc"));
        provider = this.realm.identityProviders().get("oidc");
        mapperTypes = provider.getMapperTypes();
        this.assertMapperTypes(mapperTypes, "oidc-user-attribute-idp-mapper", "oidc-role-idp-mapper", "oidc-username-idp-mapper", "oidc-advanced-group-idp-mapper", "oidc-advanced-role-idp-mapper");
        this.create(this.createRep("saml", "saml"));
        provider = this.realm.identityProviders().get("saml");
        mapperTypes = provider.getMapperTypes();
        this.assertMapperTypes(mapperTypes, "saml-user-attribute-idp-mapper", "saml-role-idp-mapper", "saml-username-idp-mapper", "saml-advanced-role-idp-mapper");
    }

    private void assertMapperTypes(Map<String, IdentityProviderMapperTypeRepresentation> mapperTypes, String ... mapperIds) {
        HashSet<String> expected = new HashSet<String>();
        expected.add("hardcoded-user-session-attribute-idp-mapper");
        expected.add("oidc-hardcoded-role-idp-mapper");
        expected.add("hardcoded-attribute-idp-mapper");
        expected.add("multi-valued-test-idp-mapper");
        expected.addAll(Arrays.asList(mapperIds));
        Assert.assertEquals((String)"mapperTypes", expected, mapperTypes.keySet());
    }

    @Test
    public void testNoExport() {
        this.create(this.createRep("keycloak-oidc", "keycloak-oidc"));
        Response response = this.realm.identityProviders().get("keycloak-oidc").export("json");
        Assert.assertEquals((String)"status", (long)204L, (long)response.getStatus());
        String body = (String)response.readEntity(String.class);
        Assert.assertNull((String)"body", (Object)body);
        response.close();
    }

    @Test
    public void testSamlImportAndExport() throws URISyntaxException, IOException, ParsingException {
        MultipartFormDataOutput form = new MultipartFormDataOutput();
        form.addFormData("providerId", (Object)"saml", MediaType.TEXT_PLAIN_TYPE);
        URL idpMeta = this.getClass().getClassLoader().getResource("admin-test/saml-idp-metadata.xml");
        byte[] content = Files.readAllBytes(Paths.get(idpMeta.toURI()));
        String body = new String(content, Charset.forName("utf-8"));
        form.addFormData("file", (Object)body, MediaType.APPLICATION_XML_TYPE, "saml-idp-metadata.xml");
        Map result = this.realm.identityProviders().importFrom(form);
        this.assertSamlImport(result, SIGNING_CERT_1, true);
        this.create(this.createRep("saml", "saml", true, result));
        IdentityProviderResource provider = this.realm.identityProviders().get("saml");
        IdentityProviderRepresentation rep = provider.toRepresentation();
        this.assertCreatedSamlIdp(rep, true);
        List providers = this.realm.identityProviders().findAll();
        Assert.assertNotNull((String)"identityProviders not null", (Object)providers);
        Assert.assertEquals((String)"identityProviders instance count", (long)1L, (long)providers.size());
        this.assertEqual(rep, (IdentityProviderRepresentation)providers.get(0));
        Response response = this.realm.identityProviders().get("saml").export("xml");
        Assert.assertEquals((long)200L, (long)response.getStatus());
        body = (String)response.readEntity(String.class);
        response.close();
        this.assertSamlExport(body);
    }

    @Test
    public void testSamlImportAndExportDisabled() throws URISyntaxException, IOException, ParsingException {
        MultipartFormDataOutput form = new MultipartFormDataOutput();
        form.addFormData("providerId", (Object)"saml", MediaType.TEXT_PLAIN_TYPE);
        URL idpMeta = this.getClass().getClassLoader().getResource("admin-test/saml-idp-metadata-disabled.xml");
        byte[] content = Files.readAllBytes(Paths.get(idpMeta.toURI()));
        String body = new String(content, Charset.forName("utf-8"));
        form.addFormData("file", (Object)body, MediaType.APPLICATION_XML_TYPE, "saml-idp-metadata-disabled.xml");
        Map result = this.realm.identityProviders().importFrom(form);
        this.assertSamlImport(result, SIGNING_CERT_1, false);
        this.create(this.createRep("saml", "saml", false, result));
        IdentityProviderResource provider = this.realm.identityProviders().get("saml");
        IdentityProviderRepresentation rep = provider.toRepresentation();
        this.assertCreatedSamlIdp(rep, false);
    }

    @Test
    public void testSamlImportAndExportMultipleSigningKeys() throws URISyntaxException, IOException, ParsingException {
        MultipartFormDataOutput form = new MultipartFormDataOutput();
        form.addFormData("providerId", (Object)"saml", MediaType.TEXT_PLAIN_TYPE);
        URL idpMeta = this.getClass().getClassLoader().getResource("admin-test/saml-idp-metadata-two-signing-certs.xml");
        byte[] content = Files.readAllBytes(Paths.get(idpMeta.toURI()));
        String body = new String(content, Charset.forName("utf-8"));
        form.addFormData("file", (Object)body, MediaType.APPLICATION_XML_TYPE, "saml-idp-metadata-two-signing-certs");
        Map result = this.realm.identityProviders().importFrom(form);
        this.assertSamlImport(result, "MIICmzCCAYMCBgFUYnC0OjANBgkqhkiG9w0BAQsFADARMQ8wDQYDVQQDDAZtYXN0ZXIwHhcNMTYwNDI5MTQzMjEzWhcNMjYwNDI5MTQzMzUzWjARMQ8wDQYDVQQDDAZtYXN0ZXIwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCN25AW1poMEZRbuMAHG58AThZmCwMV6/Gcui4mjGacRFyudgqzLjQ2rxpoW41JAtLjbjeAhuWvirUcFVcOeS3gM/ZC27qCpYighAcylZz6MYocnEe1+e8rPPk4JlID6Wv62dgu+pL/vYsQpRhvD3Y2c/ytgr5D32xF+KnzDehUy5BSyzypvu12Wq9mS5vK5tzkN37EjkhpY2ZxaXPubjDIITCAL4Q8M/m5IlacBaUZbzI4AQrHnMP1O1IH2dHSWuMiBe+xSDTco72PmuYPJKTV4wQdeBUIkYbfLc4RxVmXEvgkQgyW86EoMPxlWJpj7+mTIR+l+2thZPr/VgwTs82rAgMBAAEwDQYJKoZIhvcNAQELBQADggEBAA/Ip/Hi8RoVu5ouaFFlc5whT7ltuK8slfLGW4tM4vJXhInYwsqIRQKBNDYW/64xle3eII4u1yAH1OYRRwEs7Em1pr4QuFuTY1at+aE0sE46XDlyESI0txJjWxYoT133vM0We2pj1b2nxgU30rwjKA3whnKEfTEYT/n3JBSqNggy6l8ZGw/oPSgvPaR4+xeB1tfQFC4VrLoYKoqH6hAL530nKxL+qV8AIfL64NDEE8ankIAEDAAFe8x3CPUfXR/p4KOANKkpz8ieQaHDb1eITkAwUwjESj6UF9D1aePlhWls/HX0gujFXtWfWfrJ8CU/ogwlH8y1jgRuLjFQYZk6llc=,MIIBnDCCAQUCBgFYKXKsPTANBgkqhkiG9w0BAQsFADAUMRIwEAYDVQQDDAlzYW1sLWRlbW8wHhcNMTYxMTAzMDkwNzEwWhcNMjYxMTAzMDkwODUwWjAUMRIwEAYDVQQDDAlzYW1sLWRlbW8wgZ8wDQYJKoZIhvcNAQEBBQADgY0AMIGJAoGBAKtWsK5O0CtuBpnMvWG+HTG0vmZzujQ2o9WdheQu+BzCILcGMsbDW0YQaglpcO5JpGWWhubnckGGPHfdQ2/7nP9QwbiTK0FbGF41UqcvoaCqU1psxoV88s8IXyQCAqeyLv00yj6foqdJjxh5SZ5z+na+M7Y2OxIBVxYRAxWEnfUvAgMBAAEwDQYJKoZIhvcNAQELBQADgYEAhetvOU8TyqfZF5jpv0IcrviLl/DoFrbjByeHR+pu/vClcAOjL/u7oQELuuTfNsBI4tpexUj5G8q/YbEz0gk7idfLXrAUVcsR73oTngrhRfwUSmPrjjK0kjcRb6HL9V/+wh3R/6mEd59U08ExT8N38rhmn0CI3ehMdebReprP7U8=", true);
        this.create(this.createRep("saml", "saml", true, result));
        IdentityProviderResource provider = this.realm.identityProviders().get("saml");
        IdentityProviderRepresentation rep = provider.toRepresentation();
        this.assertCreatedSamlIdp(rep, true);
        List providers = this.realm.identityProviders().findAll();
        Assert.assertNotNull((String)"identityProviders not null", (Object)providers);
        Assert.assertEquals((String)"identityProviders instance count", (long)1L, (long)providers.size());
        this.assertEqual(rep, (IdentityProviderRepresentation)providers.get(0));
        Response response = this.realm.identityProviders().get("saml").export("xml");
        Assert.assertEquals((long)200L, (long)response.getStatus());
        body = (String)response.readEntity(String.class);
        response.close();
        this.assertSamlExport(body);
    }

    @Test
    public void testMappers() {
        this.create(this.createRep("google", "google"));
        IdentityProviderResource provider = this.realm.identityProviders().get("google");
        IdentityProviderMapperRepresentation mapper = new IdentityProviderMapperRepresentation();
        mapper.setIdentityProviderAlias("google");
        mapper.setName("my_mapper");
        mapper.setIdentityProviderMapper("oidc-hardcoded-role-idp-mapper");
        HashMap<String, String> config = new HashMap<String, String>();
        config.put("role", "offline_access");
        config.put("syncMode", IdentityProviderMapperSyncMode.INHERIT.toString());
        mapper.setConfig(config);
        Response response = provider.addMapper(mapper);
        String id = ApiUtil.getCreatedId((Response)response);
        Assert.assertNotNull((Object)id);
        response.close();
        this.assertAdminEvents.assertEvent(this.realmId, OperationType.CREATE, AdminEventPaths.identityProviderMapperPath("google", id), (Object)mapper, ResourceType.IDENTITY_PROVIDER_MAPPER);
        List mappers = provider.getMappers();
        Assert.assertEquals((String)"mappers count", (long)1L, (long)mappers.size());
        Assert.assertEquals((String)"newly created mapper id", (Object)id, (Object)((IdentityProviderMapperRepresentation)mappers.get(0)).getId());
        mapper = provider.getMapperById(id);
        Assert.assertEquals((Object)"INHERIT", ((IdentityProviderMapperRepresentation)mappers.get(0)).getConfig().get("syncMode"));
        Assert.assertNotNull((String)"mapperById not null", (Object)mapper);
        Assert.assertEquals((String)"mapper id", (Object)id, (Object)mapper.getId());
        Assert.assertNotNull((String)"mapper.config exists", (Object)mapper.getConfig());
        Assert.assertEquals((String)"config retained", (Object)"offline_access", mapper.getConfig().get("role"));
        Response error = provider.addMapper(mapper);
        Assert.assertEquals((String)"mapper unique name", (long)400L, (long)error.getStatus());
        error.close();
        mapper.getConfig().put("role", "master-realm.manage-realm");
        provider.update(id, mapper);
        this.assertAdminEvents.assertEvent(this.realmId, OperationType.UPDATE, AdminEventPaths.identityProviderMapperPath("google", id), (Object)mapper, ResourceType.IDENTITY_PROVIDER_MAPPER);
        mapper = provider.getMapperById(id);
        Assert.assertNotNull((String)"mapperById not null", (Object)mapper);
        Assert.assertEquals((String)"config changed", (Object)"master-realm.manage-realm", mapper.getConfig().get("role"));
        provider.delete(id);
        this.assertAdminEvents.assertEvent(this.realmId, OperationType.DELETE, AdminEventPaths.identityProviderMapperPath("google", id), ResourceType.IDENTITY_PROVIDER_MAPPER);
        try {
            provider.getMapperById(id);
            Assert.fail((String)"Should fail with NotFoundException");
        }
        catch (NotFoundException notFoundException) {
            // empty catch block
        }
    }

    @Test
    public void testUpdateProtocolMappers() {
        this.create(this.createRep("google2", "google"));
        IdentityProviderResource provider = this.realm.identityProviders().get("google2");
        IdentityProviderMapperRepresentation mapper = new IdentityProviderMapperRepresentation();
        mapper.setIdentityProviderAlias("google2");
        mapper.setName("my_mapper");
        mapper.setIdentityProviderMapper("oidc-hardcoded-role-idp-mapper");
        HashMap<String, String> config = new HashMap<String, String>();
        config.put("syncMode", IdentityProviderMapperSyncMode.INHERIT.toString());
        config.put("role", "");
        mapper.setConfig(config);
        Response response = provider.addMapper(mapper);
        String mapperId = ApiUtil.getCreatedId((Response)response);
        List mappers = provider.getMappers();
        org.junit.Assert.assertEquals((long)1L, (long)mappers.size());
        org.junit.Assert.assertEquals((long)1L, (long)((IdentityProviderMapperRepresentation)mappers.get(0)).getConfig().size());
        mapper = provider.getMapperById(mapperId);
        mapper.getConfig().put("role", "offline_access");
        provider.update(mapperId, mapper);
        mappers = provider.getMappers();
        org.junit.Assert.assertEquals((Object)"INHERIT", ((IdentityProviderMapperRepresentation)mappers.get(0)).getConfig().get("syncMode"));
        org.junit.Assert.assertEquals((long)1L, (long)mappers.size());
        org.junit.Assert.assertEquals((long)2L, (long)((IdentityProviderMapperRepresentation)mappers.get(0)).getConfig().size());
        org.junit.Assert.assertEquals((Object)"offline_access", ((IdentityProviderMapperRepresentation)mappers.get(0)).getConfig().get("role"));
    }

    @Test
    public void testDeleteProtocolMappersAfterDeleteIdentityProvider() {
        this.create(this.createRep("google3", "google"));
        IdentityProviderResource provider = this.realm.identityProviders().get("google3");
        IdentityProviderMapperRepresentation mapper = new IdentityProviderMapperRepresentation();
        mapper.setIdentityProviderAlias("google3");
        mapper.setName("my_mapper");
        mapper.setIdentityProviderMapper("oidc-hardcoded-role-idp-mapper");
        HashMap<String, String> config = new HashMap<String, String>();
        config.put("syncMode", IdentityProviderMapperSyncMode.INHERIT.toString());
        config.put("role", "offline_access");
        mapper.setConfig(config);
        Response response = provider.addMapper(mapper);
        List mappers = provider.getMappers();
        org.junit.Assert.assertThat((Object)mappers, (Matcher)Matchers.hasSize((int)1));
        this.assertAdminEvents.clear();
        provider.remove();
        this.assertAdminEvents.assertEvent(this.realmId, OperationType.DELETE, AdminEventPaths.identityProviderPath("google3"), ResourceType.IDENTITY_PROVIDER);
        this.create(this.createRep("google3", "google"));
        IdentityProviderResource newProvider = this.realm.identityProviders().get("google3");
        org.junit.Assert.assertThat((Object)newProvider.getMappers(), (Matcher)Matchers.empty());
    }

    @Test
    public void testInstalledIdentityProviders() {
        Response response = this.realm.identityProviders().getIdentityProviders("oidc");
        Assert.assertEquals((String)"Status", (long)200L, (long)response.getStatus());
        Map body = (Map)response.readEntity(Map.class);
        this.assertProviderInfo(body, "oidc", "OpenID Connect v1.0");
        response = this.realm.identityProviders().getIdentityProviders("keycloak-oidc");
        Assert.assertEquals((String)"Status", (long)200L, (long)response.getStatus());
        body = (Map)response.readEntity(Map.class);
        this.assertProviderInfo(body, "keycloak-oidc", "Keycloak OpenID Connect");
        response = this.realm.identityProviders().getIdentityProviders("saml");
        Assert.assertEquals((String)"Status", (long)200L, (long)response.getStatus());
        body = (Map)response.readEntity(Map.class);
        this.assertProviderInfo(body, "saml", "SAML v2.0");
        response = this.realm.identityProviders().getIdentityProviders("google");
        Assert.assertEquals((String)"Status", (long)200L, (long)response.getStatus());
        body = (Map)response.readEntity(Map.class);
        this.assertProviderInfo(body, "google", "Google");
        response = this.realm.identityProviders().getIdentityProviders("facebook");
        Assert.assertEquals((String)"Status", (long)200L, (long)response.getStatus());
        body = (Map)response.readEntity(Map.class);
        this.assertProviderInfo(body, "facebook", "Facebook");
        response = this.realm.identityProviders().getIdentityProviders("github");
        Assert.assertEquals((String)"Status", (long)200L, (long)response.getStatus());
        body = (Map)response.readEntity(Map.class);
        this.assertProviderInfo(body, "github", "GitHub");
        response = this.realm.identityProviders().getIdentityProviders("twitter");
        Assert.assertEquals((String)"Status", (long)200L, (long)response.getStatus());
        body = (Map)response.readEntity(Map.class);
        this.assertProviderInfo(body, "twitter", "Twitter");
        response = this.realm.identityProviders().getIdentityProviders("linkedin");
        Assert.assertEquals((String)"Status", (long)200L, (long)response.getStatus());
        body = (Map)response.readEntity(Map.class);
        this.assertProviderInfo(body, "linkedin", "LinkedIn");
        response = this.realm.identityProviders().getIdentityProviders("microsoft");
        Assert.assertEquals((String)"Status", (long)200L, (long)response.getStatus());
        body = (Map)response.readEntity(Map.class);
        this.assertProviderInfo(body, "microsoft", "Microsoft");
        response = this.realm.identityProviders().getIdentityProviders("stackoverflow");
        Assert.assertEquals((String)"Status", (long)200L, (long)response.getStatus());
        body = (Map)response.readEntity(Map.class);
        this.assertProviderInfo(body, "stackoverflow", "StackOverflow");
        response = this.realm.identityProviders().getIdentityProviders("nonexistent");
        Assert.assertEquals((String)"Status", (long)400L, (long)response.getStatus());
    }

    private void assertEqual(IdentityProviderRepresentation expected, IdentityProviderRepresentation actual) {
        Assert.assertNotNull((String)"expected IdentityProviderRepresentation not null", (Object)expected);
        Assert.assertNotNull((String)"actual IdentityProviderRepresentation not null", (Object)actual);
        Assert.assertEquals((String)"internalId", (Object)expected.getInternalId(), (Object)actual.getInternalId());
        Assert.assertEquals((String)"alias", (Object)expected.getAlias(), (Object)actual.getAlias());
        Assert.assertEquals((String)"providerId", (Object)expected.getProviderId(), (Object)actual.getProviderId());
        Assert.assertEquals((String)"enabled", (Object)expected.isEnabled(), (Object)actual.isEnabled());
        Assert.assertEquals((String)"firstBrokerLoginFlowAlias", (Object)expected.getFirstBrokerLoginFlowAlias(), (Object)actual.getFirstBrokerLoginFlowAlias());
        Assert.assertEquals((String)"config", (Object)expected.getConfig(), (Object)actual.getConfig());
    }

    private void assertCreatedSamlIdp(IdentityProviderRepresentation idp, boolean enabled) {
        Assert.assertNotNull((String)"IdentityProviderRepresentation not null", (Object)idp);
        Assert.assertNotNull((String)"internalId", (Object)idp.getInternalId());
        Assert.assertEquals((String)"alias", (Object)"saml", (Object)idp.getAlias());
        Assert.assertEquals((String)"providerId", (Object)"saml", (Object)idp.getProviderId());
        Assert.assertEquals((String)"enabled", (Object)enabled, (Object)idp.isEnabled());
        Assert.assertEquals((String)"firstBrokerLoginFlowAlias", (Object)"first broker login", (Object)idp.getFirstBrokerLoginFlowAlias());
        this.assertSamlConfig(idp.getConfig());
    }

    private void assertSamlConfig(Map<String, String> config) {
        org.junit.Assert.assertThat(config.keySet(), (Matcher)Matchers.containsInAnyOrder((Object[])new String[]{"validateSignature", "singleLogoutServiceUrl", "postBindingLogout", "postBindingResponse", "postBindingAuthnRequest", "singleSignOnServiceUrl", "wantAuthnRequestsSigned", "nameIDPolicyFormat", "signingCertificate", "addExtensionsElementWithKeyInfo", "loginHint", "hideOnLoginPage"}));
        org.junit.Assert.assertThat(config, (Matcher)Matchers.hasEntry((Object)"validateSignature", (Object)"true"));
        org.junit.Assert.assertThat(config, (Matcher)Matchers.hasEntry((Object)"singleLogoutServiceUrl", (Object)"http://localhost:8080/auth/realms/master/protocol/saml"));
        org.junit.Assert.assertThat(config, (Matcher)Matchers.hasEntry((Object)"postBindingResponse", (Object)"true"));
        org.junit.Assert.assertThat(config, (Matcher)Matchers.hasEntry((Object)"postBindingAuthnRequest", (Object)"true"));
        org.junit.Assert.assertThat(config, (Matcher)Matchers.hasEntry((Object)"singleSignOnServiceUrl", (Object)"http://localhost:8080/auth/realms/master/protocol/saml"));
        org.junit.Assert.assertThat(config, (Matcher)Matchers.hasEntry((Object)"wantAuthnRequestsSigned", (Object)"true"));
        org.junit.Assert.assertThat(config, (Matcher)Matchers.hasEntry((Object)"addExtensionsElementWithKeyInfo", (Object)"false"));
        org.junit.Assert.assertThat(config, (Matcher)Matchers.hasEntry((Object)"nameIDPolicyFormat", (Object)"urn:oasis:names:tc:SAML:2.0:nameid-format:persistent"));
        org.junit.Assert.assertThat(config, (Matcher)Matchers.hasEntry((Object)"hideOnLoginPage", (Object)"true"));
        org.junit.Assert.assertThat(config, (Matcher)Matchers.hasEntry((Matcher)Matchers.is((Object)"signingCertificate"), (Matcher)Matchers.notNullValue()));
    }

    private void assertSamlImport(Map<String, String> config, String expectedSigningCertificates, boolean enabled) {
        boolean enabledFromMetadata = Boolean.valueOf(config.get("enabledFromMetadata"));
        config.remove("enabledFromMetadata");
        Assert.assertEquals((Object)enabledFromMetadata, (Object)enabled);
        this.assertSamlConfig(config);
        org.junit.Assert.assertThat(config, (Matcher)Matchers.hasEntry((Object)"signingCertificate", (Object)expectedSigningCertificates));
    }

    private void assertSamlExport(String body) throws ParsingException, URISyntaxException {
        Object entBody = SAMLParser.getInstance().parse((InputStream)new ByteArrayInputStream(body.getBytes(Charset.forName("utf-8"))));
        Assert.assertEquals((String)"Parsed export type", EntityDescriptorType.class, entBody.getClass());
        EntityDescriptorType entity = (EntityDescriptorType)entBody;
        Assert.assertEquals((String)"EntityID", (Object)(OAuthClient.AUTH_SERVER_ROOT + "/realms/admin-client-test"), (Object)entity.getEntityID());
        Assert.assertNotNull((String)"ChoiceType not null", (Object)entity.getChoiceType());
        Assert.assertEquals((String)"ChoiceType.size", (long)1L, (long)entity.getChoiceType().size());
        List descriptors = ((EntityDescriptorType.EDTChoiceType)entity.getChoiceType().get(0)).getDescriptors();
        Assert.assertNotNull((String)"Descriptors not null", (Object)descriptors);
        Assert.assertEquals((String)"Descriptors.size", (long)1L, (long)descriptors.size());
        SPSSODescriptorType desc = ((EntityDescriptorType.EDTDescriptorChoiceType)descriptors.get(0)).getSpDescriptor();
        Assert.assertNotNull((String)"SPSSODescriptor not null", (Object)desc);
        Assert.assertTrue((String)"AuthnRequestsSigned", (boolean)desc.isAuthnRequestsSigned());
        HashSet<String> expected = new HashSet<String>(Arrays.asList("urn:oasis:names:tc:SAML:2.0:protocol"));
        HashSet actual = new HashSet(desc.getProtocolSupportEnumeration());
        Assert.assertEquals((String)"ProtocolSupportEnumeration", expected, actual);
        Assert.assertNotNull((String)"AssertionConsumerService not null", (Object)desc.getAssertionConsumerService());
        Assert.assertEquals((String)"AssertionConsumerService.size", (long)1L, (long)desc.getAssertionConsumerService().size());
        IndexedEndpointType endpoint = (IndexedEndpointType)desc.getAssertionConsumerService().get(0);
        Assert.assertEquals((String)"AssertionConsumerService.Location", (Object)new URI(OAuthClient.AUTH_SERVER_ROOT + "/realms/admin-client-test/broker/saml/endpoint"), (Object)endpoint.getLocation());
        Assert.assertEquals((String)"AssertionConsumerService.Binding", (Object)new URI("urn:oasis:names:tc:SAML:2.0:bindings:HTTP-POST"), (Object)endpoint.getBinding());
        Assert.assertTrue((String)"AssertionConsumerService.isDefault", (boolean)endpoint.isIsDefault());
        Assert.assertNotNull((String)"SingleLogoutService not null", (Object)desc.getSingleLogoutService());
        Assert.assertEquals((String)"SingleLogoutService.size", (long)1L, (long)desc.getSingleLogoutService().size());
        EndpointType sloEndpoint = (EndpointType)desc.getSingleLogoutService().get(0);
        Assert.assertEquals((String)"SingleLogoutService.Location", (Object)new URI(OAuthClient.AUTH_SERVER_ROOT + "/realms/admin-client-test/broker/saml/endpoint"), (Object)sloEndpoint.getLocation());
        Assert.assertEquals((String)"SingleLogoutService.Binding", (Object)new URI("urn:oasis:names:tc:SAML:2.0:bindings:HTTP-POST"), (Object)sloEndpoint.getBinding());
        Assert.assertNotNull((String)"KeyDescriptor not null", (Object)desc.getKeyDescriptor());
        Assert.assertEquals((String)"KeyDescriptor.size", (long)1L, (long)desc.getKeyDescriptor().size());
        KeyDescriptorType keyDesc = (KeyDescriptorType)desc.getKeyDescriptor().get(0);
        org.junit.Assert.assertThat((Object)keyDesc, (Matcher)Matchers.notNullValue());
        org.junit.Assert.assertThat((Object)keyDesc.getUse(), (Matcher)Matchers.equalTo((Object)KeyTypes.SIGNING));
        NodeList cert = keyDesc.getKeyInfo().getElementsByTagNameNS("http://www.w3.org/2000/09/xmldsig#", "X509Certificate");
        org.junit.Assert.assertThat((String)"KeyDescriptor.Signing.Cert existence", (Object)cert.getLength(), (Matcher)Matchers.is((Object)1));
    }

    private void assertProviderInfo(Map<String, String> info, String id, String name) {
        System.out.println(info);
        Assert.assertEquals((String)"id", (Object)id, (Object)info.get("id"));
        Assert.assertEquals((String)"name", (Object)name, (Object)info.get("name"));
    }

    @Test
    public void testSamlExportSignatureOff() throws URISyntaxException, IOException, ConfigurationException, ParsingException, ProcessingException {
        MultipartFormDataOutput form = new MultipartFormDataOutput();
        form.addFormData("providerId", (Object)"saml", MediaType.TEXT_PLAIN_TYPE);
        URL idpMeta = this.getClass().getClassLoader().getResource("admin-test/saml-idp-metadata.xml");
        byte[] content = Files.readAllBytes(Paths.get(idpMeta.toURI()));
        String body = new String(content, Charset.forName("utf-8"));
        form.addFormData("file", (Object)body, MediaType.APPLICATION_XML_TYPE, "saml-idp-metadata.xml");
        Map result = this.realm.identityProviders().importFrom(form);
        result.put("signSpMetadata", "false");
        IdentityProviderRepresentation idpRep = this.createRep("saml", "saml", true, result);
        this.create(idpRep);
        Response response = this.realm.identityProviders().get("saml").export("xml");
        Assert.assertEquals((long)200L, (long)response.getStatus());
        body = (String)response.readEntity(String.class);
        response.close();
        Document document = DocumentUtil.getDocument((String)body);
        Element signatureElement = DocumentUtil.getDirectChildElement((Element)document.getDocumentElement(), (String)JBossSAMLURIConstants.XMLDSIG_NSURI.get(), (String)"Signature");
        Assert.assertNull((Object)signatureElement);
    }

    @Test
    public void testSamlExportSignatureOn() throws URISyntaxException, IOException, ConfigurationException, ParsingException, ProcessingException {
        MultipartFormDataOutput form = new MultipartFormDataOutput();
        form.addFormData("providerId", (Object)"saml", MediaType.TEXT_PLAIN_TYPE);
        URL idpMeta = this.getClass().getClassLoader().getResource("admin-test/saml-idp-metadata.xml");
        byte[] content = Files.readAllBytes(Paths.get(idpMeta.toURI()));
        String body = new String(content, Charset.forName("utf-8"));
        form.addFormData("file", (Object)body, MediaType.APPLICATION_XML_TYPE, "saml-idp-metadata.xml");
        Map result = this.realm.identityProviders().importFrom(form);
        result.put("signSpMetadata", "true");
        IdentityProviderRepresentation idpRep = this.createRep("saml", "saml", true, result);
        this.create(idpRep);
        Response response = this.realm.identityProviders().get("saml").export("xml");
        Assert.assertEquals((long)200L, (long)response.getStatus());
        body = (String)response.readEntity(String.class);
        response.close();
        Document document = DocumentUtil.getDocument((String)body);
        Element signatureElement = DocumentUtil.getDirectChildElement((Element)document.getDocumentElement(), (String)JBossSAMLURIConstants.XMLDSIG_NSURI.get(), (String)"Signature");
        Assert.assertNotNull((Object)signatureElement);
    }
}

